<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use Doctrine\Common\Collections\Criteria;
use Exception;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\CannotRenewSubscriptionException;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\RenewSubscriptionCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use function Functional\each;
use function Functional\filter;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionsCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SymfonyStyle
     */
    private $console;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository
    ) {
        parent::__construct('subscription:renewAll');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function configure()
    {
        // do nothing for now
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->console = new SymfonyStyle($input, $output);
        $this->updateSubscriptions();
        $this->renewSubscriptions();
    }

    private function updateSubscriptions()
    {
        $subscriptions = $this->subscriptionRepository->matching(Criteria::create());

        $this->console->title(sprintf('Updating %s subscriptions', $subscriptions->count()));

        try {
            $this->console->progressStart($subscriptions->count());

            each($subscriptions, function (Subscription $subscription) {
                try {
                    $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($subscription->getId()));
                } catch (CannotRenewSubscriptionException $e) {
                    $this->console->warning($e->getMessage());
                }

                $this->console->progressAdvance(1);
            });

            $this->console->progressFinish();
            $this->console->success('Subscriptions updated');
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());
        }
    }

    /**
     * @throws Exception
     */
    private function renewSubscriptions()
    {
        $criteria = Criteria::create();
        $criteria->where(Criteria::expr()->eq('denormalizedStatus', SubscriptionStatus::expired()->getCode()));
        $criteria->where(Criteria::expr()->eq('dateCanceled', null));

        $subscriptions = $this->subscriptionRepository->matching($criteria);

        $subscriptions = filter($subscriptions, function (Subscription $subscription) {
            return $subscription->getOriginalPayment()->isDefined() &&
                $subscription->findRenewablePayment()->isDefined();
        });

        $this->console->title(sprintf('Renewing %s subscriptions', count($subscriptions)));

        try {
            $this->console->progressStart(count($subscriptions));

            $subscriptions = filter($subscriptions, function (Subscription $subscription) {
                return $subscription->getOriginalPayment()->isDefined();
            });

            each($subscriptions, function (Subscription $subscription) {
                try {
                    $this->commandBus->handle(new RenewSubscriptionCommand($subscription->getId()));
                } catch (CannotRenewSubscriptionException $e) {
                    $this->console->warning($e->getMessage());
                }
                $this->console->progressAdvance(1);
            });

            $this->console->progressFinish();
            $this->console->success('Subscriptions updated');
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());

            throw $e;
        }
    }
}
