<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\RenewSubscription;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\Subscription\RenewAmountProvider;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use League\Tactician\CommandBus;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionHandler
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var RenewAmountProvider
     */
    private $renewAmountProvider;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        InvoiceFactory $invoiceFactory,
        PaymentFactory $paymentFactory,
        SubscriptionRepository $subscriptionRepository,
        RenewAmountProvider $renewAmountProvider
    ) {
        $this->commandBus = $commandBus;
        $this->invoiceFactory = $invoiceFactory;
        $this->paymentFactory = $paymentFactory;
        $this->subscriptionRepository = $subscriptionRepository;
        $this->renewAmountProvider = $renewAmountProvider;
    }

    public function handle(RenewSubscriptionCommand $command)
    {
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());

        $renewablePayment = $subscription->findRenewablePayment()->getOrThrow(CannotRenewSubscriptionException::becauseTheSubscriptionDoesNotHaveAValidPayment());

        $newPayment = $this->paymentFactory->createRecurringPayment(
            $this->invoiceFactory->createInvoice(),
            $this->renewAmountProvider->findRenewAmount($subscription),
            $renewablePayment
        );

        $this->commandBus->handle(new ExecutePaymentCommand($newPayment->getId()));

        $subscription->createNewTerm($newPayment);
    }

    private function findSubscription(SubscriptionIdentifier $identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotRenewSubscriptionException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }
}
