<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\OpenSSL;

use IssetBV\TalosBundle\Exception\FileNotFoundException;
use IssetBV\TalosBundle\Exception\OpenSSLException;

/**
 * Class Certificate.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class Certificate extends OpenSSLThing
{
    /**
     * @var array
     */
    private $parsedData;

    /**
     * Certificate constructor.
     *
     * @param string $data
     */
    protected function __construct($data)
    {
        parent::__construct($data);
        $this->parsedData = openssl_x509_parse($data, true);
    }

    /**
     * @param string $path
     *
     * @throws FileNotFoundException if the certificate file cannot be found
     *
     * @return Certificate
     */
    public static function fromPath($path)
    {
        if (!file_exists($path)) {
            throw new FileNotFoundException($path);
        }

        return new self(file_get_contents($path));
    }

    /**
     * @param string $data
     * @param string $pspSignature
     * @param string $pspFingerprint
     *
     * @throws OpenSSLException
     *
     * @return bool
     */
    public function verify(string $data, string $pspSignature, string $pspFingerprint): bool
    {
        if (mb_strtolower($pspFingerprint) !== mb_strtolower($this->getFingerprint())) {
            // Thumbprint mismatch
            return false;
        }

        $verify = openssl_verify($data, base64_decode($pspSignature), $this->getData(), OPENSSL_ALGO_SHA512);

        if ($verify === -1) {
            //@todo investigate
            throw new OpenSSLException('openssl_verify failed: -1');
        } elseif ($verify === 0) {
            return false;
        }

        return true;
    }
}
