<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway;

use IssetBV\Json\JsonThing;
use IssetBV\TalosBundle\Gateway\Config\GatewayConfig;
use IssetBV\TalosBundle\Gateway\Exception\UnspecifiedCurrentGatewayException;
use IssetBV\TalosBundle\Gateway\Logger\GatewayLogger;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\ResponseHandler\LoggingResponseHandler;
use PhpOption\Option;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class ApiGatewayChain implements ApiGateway
{
    /**
     * @var GatewayIdProvider
     */
    private $gatewayIdProvider;

    /**
     * @var GatewayLogger
     */
    private $gatewayLogger;

    /**
     * @var ApiGateway[]
     */
    private $gateways = [];

    /**
     * @var LoggingResponseHandler
     */
    private $loggingResponseHandler;

    public function __construct(GatewayLogger $gatewayLogger = null, LoggingResponseHandler $loggingResponseHandler, GatewayIdProvider $gatewayIdProvider)
    {
        $this->gatewayIdProvider = $gatewayIdProvider;
        $this->gatewayLogger = $gatewayLogger;
        $this->loggingResponseHandler = $loggingResponseHandler;
    }

    public function configure(array $configurationData)
    {
        $this->createGateways($configurationData, $this->gatewayLogger, $this->loggingResponseHandler);
    }

    public function addGateway(ApiGateway $apiGateway, $gatewayId)
    {
        if (spl_object_hash($apiGateway) === spl_object_hash($this)) {
            throw new \InvalidArgumentException('Circular reference');
        }

        $this->gateways[$gatewayId] = $apiGateway;
    }

    public function loadServiceSpecification(Service $service): JsonThing
    {
        return $this->getCurrentGateway()->loadServiceSpecification($service);
    }

    public function send(array $requests, string $channel = 'BatchFile')
    {
        $this->getCurrentGateway()->send($requests, $channel);
    }

    private function getGateway($gatewayId): Option
    {
        return Option::fromArraysValue($this->gateways, $gatewayId);
    }

    private function getCurrentGateway(): ApiGateway
    {
        return $this->getGateway($this->gatewayIdProvider->getGatewayId())->getOrThrow(new UnspecifiedCurrentGatewayException());
    }

    /**
     * @param array $configurationData
     * @param GatewayLogger $gatewayLogger
     * @param LoggingResponseHandler $loggingResponseHandler
     */
    private function createGateways(array $configurationData, GatewayLogger $gatewayLogger, LoggingResponseHandler $loggingResponseHandler)
    {
        foreach ($configurationData as $name => $gatewayConfig) {
            $this->addGateway(
                new DefaultApiGateway(
                    new GatewayConfig(
                        $gatewayConfig['gateway_url'],
                        $gatewayConfig['private_key_path'],
                        $gatewayConfig['public_key_path'],
                        $gatewayConfig['request_engine_public_key_path'],
                        $gatewayConfig['sender_code'],
                        false
                    ),
                    $gatewayLogger,
                    $loggingResponseHandler
                ),
                $name
            );
        }
    }
}
