<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\VerifyPaymentStatus;

use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizePaymentCommand;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Entity\SubscriptionTerm;
use League\Tactician\CommandBus;
use function \Functional\each;
use function \Functional\filter;
use function \Functional\flat_map;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class VerifyPaymentStatusHandler
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    /**
     * VerifyPaymentStatusHandler constructor.
     *
     * @param CommandBus $commandBus
     * @param SubscriptionRepository $subscriptionRepository
     */
    public function __construct(CommandBus $commandBus, SubscriptionRepository $subscriptionRepository)
    {
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    /**
     * @param VerifyPaymentStatusCommand $command
     */
    public function handle(VerifyPaymentStatusCommand $command)
    {
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());
        $subscriptionTerms = $subscription->getSubscriptionTerms();

        // Filter terms less then 60 days old
        $subscriptionTerms = filter($subscriptionTerms, function (SubscriptionTerm $subscriptionTerm) {
            $diff = (new \DateTime())->diff($subscriptionTerm->getDateTime());

            return $diff->days < 60;
        });

        $payments = flat_map($subscriptionTerms, function (SubscriptionTerm $subscriptionTerm) {
            return $subscriptionTerm->getPayment()->getOrElse([]);
        });

        each($payments, [$this, 'refreshPayment']);

        //$this->findSubscription()
        //    |> map($$, fn($subscription) => $subscription->getTerms())
        //    |> filter($$, lessThen60DaysOld)
        //    |> flat_map($$, fn($subscriptionTerm) => $subscriptionTerm->getPayment()->getOrElse([]))
        //    |> each($$, [$this, ['refreshPayment']]);
    }

    public function refreshPayment(Payment $payment)
    {
        $this->commandBus->handle(new SynchronizePaymentCommand($payment->getId()));
    }

    private function findSubscription(SubscriptionIdentifier $subscriptionIdentifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($subscriptionIdentifier)
            ->getOrThrow(CannotVerifyPaymentStatusException::becauseTheSubscriptionDoesNotExist($subscriptionIdentifier->asString()));
    }
}
