<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Config;

use IssetBV\TalosBundle\OpenSSL\Certificate;
use IssetBV\TalosBundle\OpenSSL\PrivateKey;
use League\Uri\Schemes\Http;

/**
 * Class GatewayConfig.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class GatewayConfig
{
    /**
     * @var string
     */
    private $apiUrl;

    /**
     * @var bool
     */
    private $debugMode;

    /**
     * @var PrivateKey
     */
    private $privateKey;

    /**
     * @var Certificate
     */
    private $publicKey;

    /**
     * @var Certificate
     */
    private $requestEnginePublicKey;

    /**
     * @var string
     */
    private $senderCode;

    /**
     * @var Http
     */
    private $url;

    /**
     * GatewayConfig constructor.
     *
     * @param string $apiUrl
     * @param string $privateKeyPath
     * @param string $publicKeyPath
     * @param string $requestEnginePublicKeyPath
     * @param string $senderCode
     * @param bool $debug
     */
    public function __construct(string $apiUrl, string $privateKeyPath, string $publicKeyPath, string $requestEnginePublicKeyPath, string $senderCode, bool $debug)
    {
        $this->url = Http::createFromString($apiUrl);
        $this->apiUrl = $apiUrl;
        $this->privateKey = PrivateKey::fromPath($privateKeyPath);
        $this->publicKey = Certificate::fromPath($publicKeyPath);
        $this->requestEnginePublicKey = Certificate::fromPath($requestEnginePublicKeyPath);
        $this->senderCode = $senderCode;
        $this->debugMode = $debug;
    }

    /**
     * @return string
     */
    public function getApiUrl(): string
    {
        return $this->apiUrl;
    }

    /**
     * @return PrivateKey
     */
    public function getPrivateKey(): PrivateKey
    {
        return $this->privateKey;
    }

    /**
     * @return Certificate
     */
    public function getPublicKey(): Certificate
    {
        return $this->publicKey;
    }

    /**
     * @return Certificate
     */
    public function getRequestEnginePublicKey(): Certificate
    {
        return $this->requestEnginePublicKey;
    }

    /**
     * @return string
     */
    public function getSenderCode(): string
    {
        return $this->senderCode;
    }

    /**
     * @return Http
     */
    public function getUrl(): Http
    {
        return $this->url;
    }

    /**
     * @return bool
     */
    public function isDebugMode(): bool
    {
        return $this->debugMode;
    }
}
