<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\SynchronizePayment;

use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\CannotSynchronizePaymentException;
use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizePaymentCommand;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\Mapper\Exception\MapperException;
use IssetBV\TalosBundle\ResponseHandler\PaymentResponseHandler;

/**
 * Class TalosSynchronizePaymentHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizePaymentHandler
{
    const ACTION_NAME = 'Status';

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var PaymentResponseHandler
     */
    private $paymentResponseHandler;

    /**
     * TalosSynchronizePaymentHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param PaymentResponseHandler $paymentResponseHandler
     * @param PaymentRepository $paymentRepository
     */
    public function __construct(
        ApiGateway $gateway,
        PaymentResponseHandler $paymentResponseHandler,
        PaymentRepository $paymentRepository
    ) {
        $this->gateway = $gateway;
        $this->paymentResponseHandler = $paymentResponseHandler;
        $this->paymentRepository = $paymentRepository;
    }

    /**
     * @param SynchronizePaymentCommand $command
     *
     * @throws CannotSynchronizePaymentException
     */
    public function handle(SynchronizePaymentCommand $command)
    {
        // We load the payment from the DB to ensure it exists. Better to fail early
        $this->findPayment($command->getPaymentIdentifier()); // throws

        $request = new Request(ClientData::empty(), [
            new Service('General', self::ACTION_NAME, [
                new SingleField('ObjectType', 'Payment'),
//                new SingleField('ObjectIdentifier', ), //@todo this is the talos ID that we don't need to use AFAIK
                new SingleField('ObjectReference', (string) $command->getPaymentIdentifier()),
            ]),
        ]);

        $request->addResponseHandler($this->paymentResponseHandler);

        try {
            $this->gateway->send([$request]);
        } catch (MapperException $e) {
            throw CannotSynchronizePaymentException::becauseTheGatewayResponseCouldNotBeMapped($e);
        }
    }

    protected function findPayment(LocalPaymentIdentifier $paymentIdentifier): Payment
    {
        return $this->paymentRepository->optionallyFind($paymentIdentifier)
            ->getOrThrow(CannotSynchronizePaymentException::becauseThePaymentCouldNotBeFound($paymentIdentifier));
    }
}
