<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\CreateWallet;

use IssetBV\PaymentBundle\CommandBus\CreateWallet\CannotCreateWalletException;
use IssetBV\PaymentBundle\CommandBus\CreateWallet\CreateWalletCommand;
use IssetBV\PaymentBundle\Domain\Repository\WalletOwnerRepository;
use IssetBV\PaymentBundle\Domain\Repository\WalletTypeRepository;
use IssetBV\PaymentBundle\Generator\ReferenceGenerator;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\WalletResponseHandler;

/**
 * Class CreateWalletHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class CreateWalletHandler
{
    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var WalletOwnerRepository
     */
    private $walletOwnerRepository;

    /**
     * @var WalletResponseHandler
     */
    private $walletResponseHandler;

    /**
     * @var WalletTypeRepository
     */
    private $walletTypeRepository;

    /**
     * CreateWalletHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param WalletOwnerRepository $walletOwnerRepository
     * @param WalletTypeRepository $walletTypeRepository
     * @param WalletResponseHandler $walletResponseHandler
     */
    public function __construct(
        ApiGateway $gateway,
        WalletOwnerRepository $walletOwnerRepository,
        WalletTypeRepository $walletTypeRepository,
        WalletResponseHandler $walletResponseHandler
    ) {
        $this->gateway = $gateway;
        $this->walletOwnerRepository = $walletOwnerRepository;
        $this->walletTypeRepository = $walletTypeRepository;
        $this->walletResponseHandler = $walletResponseHandler;
    }

    /**
     * @param CreateWalletCommand $command
     *
     * @throws CannotCreateWalletException
     */
    public function handle(CreateWalletCommand $command)
    {
        $walletOwner = $this->walletOwnerRepository->optionallyFind($command->getWalletOwnerIdentifier())
            ->getOrThrow(new CannotCreateWalletException('Wallet owner with id ' . $command->getWalletOwnerIdentifier() . ' does not exist'));

        if (count($walletOwner->getWallets()) > 0) {
            throw new CannotCreateWalletException('User already has a wallet');
        }

        $walletType = $this->walletTypeRepository->optionallyFind($command->getWalletTypeIdentifier())
            ->getOrThrow(new CannotCreateWalletException('You must specify a WalletType option'));

        $request = new Request(
            ClientData::empty(), [
                new Service(
                    'WalletSystem', 'Wallet', [
                        new SingleField('TypeKey', $walletType->getId()),
                        new SingleField('StatusCode', 'Active'),
                        new SingleField('Reference', ReferenceGenerator::generateWalletReference($walletOwner)),
                        new SingleField('Description', $command->getDescription()),
                    ]
                ),
            ]
        );

        $request->addResponseHandler($this->walletResponseHandler);

        $this->gateway->send([$request]);
    }
}
