<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Factory;

use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Identifier\UuidPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Invoice\Invoice;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use Money\Money;
use PhpOption\Option;
use Ramsey\Uuid\Uuid;

/**
 * Class SimplePayment.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SimplePayment implements Payment
{
    /**
     * @var LocalPaymentIdentifier
     */
    private $id;

    /**
     * @var Money
     */
    private $amount;

    /**
     * @var Invoice
     */
    private $invoice;

    /**
     * @var PaymentIssuer
     */
    private $paymentIssuer;

    /**
     * @var PaymentMethod
     */
    private $paymentMethod;

    /**
     * @var PaymentType
     */
    private $recurrentType;

    /**
     * @var string
     */
    private $reference;

    /**
     * @var string|null
     */
    private $remoteIdentifier;

    /**
     * @var string|null
     */
    private $remoteStatus;

    /**
     * SimplePayment constructor.
     *
     * @param Invoice $invoice
     * @param Money $money
     * @param PaymentMethod $paymentMethod
     * @param PaymentIssuer|null $issuer
     * @param PaymentType $type
     */
    public function __construct(Invoice $invoice, Money $money, PaymentMethod $paymentMethod, PaymentIssuer $issuer = null, PaymentType $type)
    {
        $this->id = new UuidPaymentIdentifier(Uuid::uuid4());
        $this->amount = $money;
        $this->invoice = $invoice;
        $this->paymentIssuer = $issuer;
        $this->paymentMethod = $paymentMethod;
        $this->recurrentType = $type;
        $this->reference = (string) $this->id;
        $this->remoteIdentifier = null;
        $this->remoteStatus = null;
    }

    /**
     * @return LocalPaymentIdentifier
     */
    public function getId(): LocalPaymentIdentifier
    {
        return $this->id;
    }

    /**
     * @return Money
     */
    public function getAmount(): Money
    {
        return $this->amount;
    }

    /**
     * @return Invoice
     */
    public function getInvoice(): Invoice
    {
        return $this->invoice;
    }

    /**
     * @return string
     */
    public function getReference(): string
    {
        return $this->reference;
    }

    /**
     * @return PaymentMethod
     */
    public function getPaymentMethod(): PaymentMethod
    {
        return $this->paymentMethod;
    }

    /**
     * @return Option
     */
    public function getPaymentIssuer(): Option
    {
        return Option::ensure($this->paymentIssuer);
    }

    /**
     * Returns the recurrent type of this payment. Currently this is a string "First", "OneOff", "Recurring".
     *
     * @return string
     */
    public function getRecurrentType(): string
    {
        if ($this->recurrentType->isOneOff()) {
            return 'OneOff';
        } elseif ($this->recurrentType->isRecurring()) {
            return 'Recurring';
        }

        return 'First';
    }

    /**
     * Returns true if this payment is usable for auto topup profiles.
     *
     * @return bool
     */
    public function isUsableForAutoTopup(): bool
    {
        return false;
    }

    /**
     * Returns the remote identifier of this object.
     *
     * @return Option
     */
    public function getRemoteIdentifier(): Option
    {
        return Option::ensure($this->remoteIdentifier);
    }

    /**
     * @param string $remoteIdentifier
     */
    public function setRemoteIdentifier(string $remoteIdentifier)
    {
        $this->remoteIdentifier = $remoteIdentifier;
    }

    /**
     * @param string $remoteStatus
     */
    public function setRemoteStatus(string $remoteStatus)
    {
        $this->remoteStatus = $remoteStatus;
    }

    /**
     * @return string
     */
    public function getRemoteStatus(): string
    {
        return $this->remoteStatus;
    }

    /**
     * @return PaymentType
     */
    public function getType(): PaymentType
    {
        return $this->getType();
    }
}
