<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizePaymentCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Entity\Subscription;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class UpdateSubscriptionCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SymfonyStyle
     */
    private $console;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository
    ) {
        parent::__construct('subscription:update');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function configure()
    {
        $this->addArgument('subscription', InputArgument::REQUIRED, 'The identifier of the subscription that you wish to update', null);
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->console = new SymfonyStyle($input, $output);

        /** @var Subscription $subscription */
        $subscription = $this->subscriptionRepository->optionallyFind(new SubscriptionIdentifier($input->getArgument('subscription')))
            ->getOrThrow(new \RuntimeException('Subscription could not be found'));

        $paymentCount = $subscription->getPayments()->count();

        $this->console->section(sprintf('Updating the status of %d payments', $paymentCount));
        $this->console->progressStart($paymentCount);

        foreach ($subscription->getPayments() as $payment) {
            $this->commandBus->handle(new SynchronizePaymentCommand($payment->getId()));
            $this->console->progressAdvance(1);
        }

        $this->console->progressFinish();
    }
}
