<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Doctrine;

use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\ConversionException;
use Doctrine\DBAL\Types\Type;
use Money\Currency;
use Money\Money;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class MoneyType extends Type
{
    const NAME = 'money';

    /**
     * @param array $fieldDeclaration
     * @param AbstractPlatform $platform
     *
     * @return string
     */
    public function getSqlDeclaration(array $fieldDeclaration, AbstractPlatform $platform)
    {
        return $platform->getVarcharTypeDeclarationSQL($fieldDeclaration);
    }

    /**
     * @param AbstractPlatform $platform
     *
     * @return bool
     */
    public function requiresSQLCommentHint(AbstractPlatform $platform)
    {
        return true;
    }

    /**
     * @param mixed $value
     * @param AbstractPlatform $platform
     *
     * @return Money|null
     */
    public function convertToPHPValue($value, AbstractPlatform $platform)
    {
        if ($value === null) {
            return null;
        }

        list($currency, $amount) = explode(' ', $value, 2);

        return new Money((int) $amount, new Currency($currency));
    }

    /**
     * @param mixed $value
     * @param AbstractPlatform $platform
     *
     * @throws ConversionException
     *
     * @return null|string
     */
    public function convertToDatabaseValue($value, AbstractPlatform $platform)
    {
        if (empty($value)) { // such cancerous
            return null;
        }

        if ($value instanceof Money) {
            return (string) $value->getCurrency() . ' ' . $value->getAmount();
        }

        throw ConversionException::conversionFailed($value, self::NAME);
    }

    /**
     * @return string
     */
    public function getName()
    {
        return self::NAME;
    }
}
