<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler;

use IssetBV\PaymentBundle\Domain\Repository\MutationRepository;
use IssetBV\TalosBundle\Entity\TalosMutation;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\Mapper\MutationMapper;

/**
 * Class CreateMutationResponseHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class MutationResponseHandler implements ResponseHandler
{
    const REQUEST_OBJECT_TYPE = 'WalletMutation';

    /**
     * @var callable
     */
    private $mutationConsumer;

    /**
     * @var MutationMapper
     */
    private $mutationMapper;

    /**
     * @var MutationRepository
     */
    private $mutationRepository;

    /**
     * CreateMutationResponseHandler constructor.
     *
     * @param MutationRepository $mutationRepository
     * @param MutationMapper $mutationMapper
     */
    public function __construct(MutationRepository $mutationRepository, MutationMapper $mutationMapper)
    {
        $this->mutationRepository = $mutationRepository;
        $this->mutationMapper = $mutationMapper;
    }

    /**
     * @param callable $mutationConsumer
     */
    public function setMutationConsumer(callable $mutationConsumer)
    {
        $this->mutationConsumer = $mutationConsumer;
    }

    /**
     * @param Response $response
     */
    public function handleResponse(Response $response)
    {
        foreach ($response->getRequestObjects() as $requestObject) {
            if ($requestObject->getType() === self::REQUEST_OBJECT_TYPE) {
                // We use the remote identifier to create or find a local copy
                $identifier = $requestObject->getIdentifier();

                // Find the existing mutation or create a new one
                $mutation = $this->mutationRepository
                    ->findWithRemoteIdentifier($identifier)
                    ->getOrCall(function () use ($identifier) {
                        $mutation = new TalosMutation($identifier);
                        $this->mutationRepository->save($mutation);

                        return $mutation;
                    });

                $this->mutationMapper->map($requestObject, $mutation);

                if (is_callable($this->mutationConsumer)) {
                    ($this->mutationConsumer)($mutation);
                }
            }
        }
    }
}
