<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler;

use IssetBV\PaymentBundle\Domain\Repository\AutoTopupProfileRepository;
use IssetBV\TalosBundle\Entity\TalosAutoTopupProfile;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\Storage\EntityStore;
use Money\Parser\DecimalMoneyParser;
use function Functional\each;

/**
 * Class CreateAutoTopupProfileResponseHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class AutoTopupProfileResponseHandler implements ResponseHandler
{
    /**
     * @var AutoTopupProfileRepository
     */
    private $autoTopupProfileRepository;

    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var DecimalMoneyParser
     */
    private $moneyParser;

    /**
     * AutoTopupProfileResponseHandler constructor.
     *
     * @param AutoTopupProfileRepository $autoTopupProfileRepository
     * @param EntityStore $entityStore
     * @param DecimalMoneyParser $moneyParser
     */
    public function __construct(
        AutoTopupProfileRepository $autoTopupProfileRepository,
        EntityStore $entityStore,
        DecimalMoneyParser $moneyParser
    ) {
        $this->autoTopupProfileRepository = $autoTopupProfileRepository;
        $this->entityStore = $entityStore;
        $this->moneyParser = $moneyParser;
    }

    /**
     * @param Response $response
     */
    public function handleResponse(Response $response)
    {
        each($response->getRequestObjectsWithType('WalletAutoTopupProfile'), [$this, 'processAutoTopupProfile']);
    }

    /**
     * @param RequestObject $requestObject
     */
    public function processAutoTopupProfile(RequestObject $requestObject)
    {
        $autoTopupProfile = $this->autoTopupProfileRepository->optionallyFind($requestObject->getIdentifier());

        $toMoney = function ($value) {
            $this->moneyParser->parse($value, 'EUR');
        };

        $autoTopupProfile->getOrCall(function () use ($requestObject, $toMoney) {
            $autoTopupProfile = new TalosAutoTopupProfile(
                $requestObject->getIdentifier(),
                $requestObject->getSingleFieldValue('Cooldown')->map('intval')->getOrElse(null),
                $requestObject->getSingleFieldValue('LimitPeriod')->map('intval')->getOrElse(null),
                $requestObject->getSingleFieldValue('LimitAmount')->map($toMoney)->getOrElse(null),
                $requestObject->getSingleFieldValue('ActivationBalance')->map($toMoney)->getOrElse(null)
            );

            $this->entityStore->persist($autoTopupProfile);
        });
    }
}
