<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use Assert\Assertion;
use DateInterval;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription as SubscriptionInterface;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\TalosBundle\Storage\CreatedUpdatedFields;
use PhpOption\Option;
use function Functional\map;
use function Functional\maximum;

/**
 * @author Tim Fennis <tim@isset.nl>
 *
 * @ORM\Entity(repositoryClass="IssetBV\PaymentBundle\Repository\DoctrineSubscriptionRepository")
 * @ORM\Table(name="subscriptions")
 * @ORM\HasLifecycleCallbacks()
 */
class Subscription implements SubscriptionInterface
{
    use CreatedUpdatedFields;

    /**
     * @var int
     * @ORM\Id()
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var int
     * @ORM\Column(name="denormalized_status", type="integer", nullable=false)
     */
    private $denormalizedStatus;

    /**
     * @var DateTime
     * @ORM\Column(name="start_date", type="datetime", nullable=true)
     */
    private $startDate;

    /**
     * @var DateTime
     * @ORM\Column(name="date_canceled", type="datetime", nullable=true)
     */
    private $dateCanceled;

    /**
     * @var Interval
     * @ORM\Embedded(class="Interval")
     */
    private $paymentInterval;

    /**
     * @var ArrayCollection
     * @ORM\OneToMany(targetEntity="IssetBV\PaymentBundle\Entity\SubscriptionTerm", mappedBy="subscription", cascade={"all"})
     */
    private $subscriptionTerms;

    /**
     * @var Payment
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     * @ORM\JoinColumn(name="original_payment_id")
     */
    private $originalPayment;

    public function __construct(DateInterval $paymentInterval, DateTime $startDate, Payment $initialPayment, DateInterval $initialInterval = null)
    {
        $this->initCrudFields();
        $this->paymentInterval = Interval::createFromDateInterval($paymentInterval);
        $this->startDate = $startDate;
        $this->subscriptionTerms = new ArrayCollection();

        // Set the initial status to pending
        $this->updateDenormalizedStatus(SubscriptionStatus::pending());

        $this->createFirstTerm(
            $initialPayment,
            $initialInterval
                ? Interval::createFromDateInterval($initialInterval)
                : $this->paymentInterval
        );
    }

    public function getOriginalPayment(): Payment
    {
        return $this->originalPayment;
    }

    /**
     * @param Payment $payment
     * @param Interval $interval
     *
     * @throws
     */
    public function createFirstTerm(Payment $payment, Interval $interval)
    {
        Assertion::null($this->originalPayment);
        $this->originalPayment = $payment;
        $this->subscriptionTerms->add(new SubscriptionTerm($this, $this->originalPayment, $interval, false));
    }

    /**
     * Returns a collection compatible with `Collection` and `Selectable`.
     *
     * @return Collection|Payment[]
     */
    public function getPayments()
    {
        return $this->getSubscriptionTerms()
            ->map(function (SubscriptionTerm $subscriptionPayment) {
                return $subscriptionPayment->getPayment();
            });
    }

    /**
     * @deprecated Use getSubscriptionTerms instead
     *
     * @return ArrayCollection
     */
    public function getSubscriptionPayments()
    {
        return $this->subscriptionTerms;
    }

    /**
     * @return SubscriptionTerm[]|ArrayCollection
     */
    public function getSubscriptionTerms()
    {
        return $this->subscriptionTerms;
    }

    public function getId(): SubscriptionIdentifier
    {
        return new SubscriptionIdentifier($this->id);
    }

    public function getStartDate(): Option
    {
        return Option::ensure($this->startDate);
    }

    /**
     * @throws InvalidPaymentIntervalException if the database has somehow gotten corrupted
     *
     * @return DateInterval
     */
    public function getPaymentInterval(): DateInterval
    {
        return $this->paymentInterval->getDateInterval()
            ->getOrThrow(new InvalidPaymentIntervalException());
    }

    public function getDateValidUntil(): Option
    {
        $maxTimestamp = maximum(map($this->getSubscriptionTerms(), function (SubscriptionTerm $subscriptionPayment) {
            $date = clone $subscriptionPayment->getDateTime();

            return $date->add($subscriptionPayment->getTermInterval())->getTimestamp();
        }));

        return Option::ensure($maxTimestamp)
            ->map(function (int $maxTimestamp) {
                return new DateTime("@$maxTimestamp");
            });
    }

    public function cancel(): bool
    {
        return $this->dateCanceled === null
            ? (bool) ($this->dateCanceled = new DateTime())
            : false;
    }

    public function updateDenormalizedStatus(SubscriptionStatus $subscriptionStatus)
    {
        $this->denormalizedStatus = $subscriptionStatus->getCode();
    }

    public function getStatus(): SubscriptionStatus
    {
        return SubscriptionStatus::safelyFromInteger($this->denormalizedStatus);
    }

    public function getDateCanceled(): Option
    {
        return Option::ensure($this->dateCanceled);
    }
}
