<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\ExecutePayment;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\CannotExecutePaymentException;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\PaymentToWallet;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Entity\TalosPaymentIssuer;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\PaymentResponseHandler;
use League\Uri\Schemes\Http;
use Money\Formatter\DecimalMoneyFormatter;
use Money\MoneyFormatter;

/**
 * This class handles `ExecutePaymentCommand`s. By sending the necessary information to the payment gateway.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class ExecutePaymentHandler
{
    /**
     * @var MoneyFormatter
     */
    private $formatter;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var PaymentResponseHandler
     */
    private $paymentResponseHandler;

    /**
     * ExecutePaymentHandler constructor.
     *
     * @param DecimalMoneyFormatter $formatter
     * @param ApiGateway $gateway
     * @param PaymentRepository $paymentRepository
     * @param PaymentResponseHandler $paymentResponseHandler
     */
    public function __construct(DecimalMoneyFormatter $formatter, ApiGateway $gateway, PaymentRepository $paymentRepository, PaymentResponseHandler $paymentResponseHandler)
    {
        $this->formatter = $formatter;
        $this->gateway = $gateway;
        $this->paymentRepository = $paymentRepository;
        $this->paymentResponseHandler = $paymentResponseHandler;
    }

    /**
     * @param ExecutePaymentCommand $command
     *
     * @throws CannotExecutePaymentException
     */
    public function handle(ExecutePaymentCommand $command)
    {
        $paymentMethod = $this->findPaymentMethod($command);

        if ($paymentMethod->getServiceName() !== 'iDEAL') {
            throw new CannotExecutePaymentException(self::class . ' cannot handle ' . $paymentMethod->getServiceName() . ' at the moment');
        }

        $payment = $this->findPayment($command->getPaymentId());

        $request = $this->createIdealPaymentRequest(
            $payment,
            Http::createFromString($command->getReturnUrl()),
            $command->getClientData()->getOrElse(ClientData::empty())
        );

        $this->gateway->send([$request], 'Web');
    }

    /**
     * @param ExecutablePayment $payment
     * @param Http $returnUrl
     * @param ClientData $clientData
     *
     * @throws CannotExecutePaymentException
     *
     * @return Request
     */
    private function createIdealPaymentRequest(ExecutablePayment $payment, Http $returnUrl, ClientData $clientData): Request
    {
        $services[] = $this->createIdealPart($payment);

        if ($payment instanceof PaymentToWallet) {
            $services[] = $this->createWalletTransferPart($payment);
        }

        $request = new Request($clientData, $services, (string) $returnUrl);
        $request->addResponseHandler($this->paymentResponseHandler);

        return $request;
    }

    private function createIdealPart(ExecutablePayment $payment): Service
    {
        /** @var TalosPaymentIssuer $issuer */
        $issuer = $payment->getPaymentIssuer()
            ->getOrThrow(CannotExecutePaymentException::becauseNoIssuerWasSet());

        if (false === $issuer instanceof TalosPaymentIssuer) {
            throw new CannotExecutePaymentException('Issuer must by of type ' . TalosPaymentIssuer::class . '. ' . get_class($issuer) . ' given');
        }

        return new Service('iDEAL', 'Pay', [
            new GroupField('Payment', [
                new SingleField('InvoiceNumber', $payment->getInvoice()->getNumber()->getAsString()),
                new SingleField('Reference', $payment->getReference()),
                new SingleField('Description', $payment->getDescription()->getOrElse('')),
                new SingleField('Currency', $payment->getAmount()->getCurrency()->getCode()),
                new SingleField('RecurrentType', $payment->getRecurrentType()),
                new SingleField('AmountDebit', $this->formatter->format($payment->getAmount())),
            ]),
            new SingleField('Issuer', $issuer->getCode()),
        ]);
    }

    private function findPayment(LocalPaymentIdentifier $paymentId): ExecutablePayment
    {
        return $this->paymentRepository
            ->optionallyFind($paymentId)
            ->getOrThrow(CannotExecutePaymentException::becauseThePaymentDoesNotExist($paymentId));
    }

    private function findPaymentMethod(ExecutePaymentCommand $command): TalosPaymentMethod
    {
        $paymentMethod = $this
            ->findPayment($command->getPaymentId())
            ->getPaymentMethod();

        if ($paymentMethod instanceof TalosPaymentMethod) {
            return $paymentMethod;
        }

        throw new CannotExecutePaymentException(self::class . ' cannot handle instances of ' . get_class($paymentMethod));
    }

    private function createWalletTransferPart(PaymentToWallet $payment): Service
    {
        return new Service('WalletSystem', 'Transfer', [
            new SingleField('SourceWalletKey', $payment->getPaymentMethod()->getWallet()->getIdentifier()),
            new SingleField('DestinationWalletKey', $payment->getWallet()->getIdentifier()),
            new SingleField('Amount', $this->formatter->format($payment->getAmount())),
            new SingleField('Description', 'Payment for Friday'),
            new GroupField('Payment', [
//                    new SingleField('PaymentKey', 'PAYREF123'),
                new SingleField('WaitForPaymentSuccess', 'true'), // because ideal is fast as shit
            ]),
        ]);
    }
}
