<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use DateInterval;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription as SubscriptionInterface;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\TalosBundle\Storage\CreatedUpdatedFields;
use PhpOption\Option;
use function Functional\map;
use function Functional\maximum;

/**
 * @author Tim Fennis <tim@isset.nl>
 *
 * @ORM\Entity(repositoryClass="IssetBV\PaymentBundle\Repository\DoctrineSubscriptionRepository")
 * @ORM\Table(name="subscriptions")
 * @ORM\HasLifecycleCallbacks()
 */
class Subscription implements SubscriptionInterface
{
    use CreatedUpdatedFields;

    /**
     * @var int
     * @ORM\Id()
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var int
     * @ORM\Column(name="denormalized_status", type="integer", nullable=false)
     */
    private $denormalizedStatus;

    /**
     * @var DateTime
     * @ORM\Column(name="start_date", type="datetime", nullable=true)
     */
    private $startDate;

    /**
     * @var DateTime
     * @ORM\Column(name="date_canceled", type="datetime", nullable=true)
     */
    private $dateCanceled;

    /**
     * @var Interval
     * @ORM\Embedded(class="Interval")
     */
    private $paymentInterval;

    /**
     * @var ArrayCollection
     * @ORM\OneToMany(targetEntity="IssetBV\PaymentBundle\Entity\SubscriptionPayment", mappedBy="subscription", cascade={"all"})
     */
    private $subscriptionPayments;

    /**
     * @var Payment
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     * @ORM\JoinColumn(name="original_payment_id")
     */
    private $originalPayment;

    public function __construct(DateInterval $paymentInterval, Payment $originalPayment, DateTime $startDate)
    {
        $this->initCrudFields();
        $this->paymentInterval = Interval::createFromDateInterval($paymentInterval);
        $this->startDate = $startDate;

        // Set the initial status to pending
        $this->updateDenormalizedStatus(SubscriptionStatus::pending());

        // Add the original payment to the collection of payments
        $this->subscriptionPayments = new ArrayCollection();
        $this->subscriptionPayments->add(new SubscriptionPayment($this, $originalPayment));
        $this->originalPayment = $originalPayment;
    }

    public function getOriginalPayment(): Payment
    {
        return $this->originalPayment;
    }

    /**
     * Returns a collection compatible with `Collection` and `Selectable`.
     *
     * @return Collection|Payment[]
     */
    public function getPayments()
    {
        return $this->subscriptionPayments
            ->map(function (SubscriptionPayment $subscriptionPayment) {
                return $subscriptionPayment->getPayment();
            });
    }

    public function getSubscriptionPayments()
    {
        return $this->subscriptionPayments;
    }

    public function getId(): SubscriptionIdentifier
    {
        return new SubscriptionIdentifier($this->id);
    }

    public function getStartDate(): Option
    {
        return Option::ensure($this->startDate);
    }

    /**
     * @throws InvalidPaymentIntervalException if the database has somehow gotten corrupted
     *
     * @return DateInterval
     */
    public function getPaymentInterval(): DateInterval
    {
        return $this->paymentInterval->getDateInterval()
            ->getOrThrow(new InvalidPaymentIntervalException());
    }

    public function getDateValidUntil(): Option
    {
        $maxTimestamp = maximum(map($this->getSubscriptionPayments(), function (SubscriptionPayment $subscriptionPayment) {
            return $subscriptionPayment->getDateTime()->getTimestamp();
        }));

        return Option::ensure($maxTimestamp)
            ->map(function (int $maxTimestamp) {
                $time = new DateTime("@$maxTimestamp");
                return $time->add($this->getPaymentInterval());
            });
    }

    public function cancel(): bool
    {
        return $this->dateCanceled === null && $this->getStatus()->isValid()
            ? (bool) ($this->dateCanceled = new DateTime())
            : false;
    }

    public function updateDenormalizedStatus(SubscriptionStatus $subscriptionStatus)
    {
        $this->denormalizedStatus = $subscriptionStatus->getCode();
    }

    public function getStatus(): SubscriptionStatus
    {
        return SubscriptionStatus::safelyFromInteger($this->denormalizedStatus);
    }

    public function getDateCanceled(): Option
    {
        return Option::ensure($this->dateCanceled);
    }
}
