<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus;

use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Entity\SubscriptionPayment;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use function Functional\filter;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class DenormalizeSubscriptionStatusHandler
{
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(SubscriptionRepository $subscriptionRepository, EventDispatcher $eventDispatcher)
    {
        $this->subscriptionRepository = $subscriptionRepository;
        $this->eventDispatcher = $eventDispatcher;
    }

    public function handle(DenormalizeSubscriptionStatusCommand $command)
    {
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());
        $originalStatus = $subscription->getStatus();
        $subscription->updateDenormalizedStatus($this->determineStatus($subscription));
        $newStatus = $subscription->getStatus();

        // delta
        if (false === $originalStatus->equals($newStatus)) {
            $this->eventDispatcher->dispatch(
                SubscriptionStatusChangedEvent::name(),
                new SubscriptionStatusChangedEvent($subscription->getId(), $originalStatus, $newStatus)
            );
        }
    }

    protected function determineStatus(Subscription $subscription): SubscriptionStatus
    {
        $links = $subscription->getSubscriptionPayments();

        $activeLinks = filter($links, function (SubscriptionPayment $subscriptionPayment) {
            return $subscriptionPayment->isActive();
        });

        $validLinks = filter($activeLinks, function (SubscriptionPayment $subscriptionPayment) {
            return $subscriptionPayment->getPayment()->getRemoteStatus() === 'Success';
        });

        if ($subscription->getDateCanceled()->isDefined()) {
            return SubscriptionStatus::canceled();
        }

        if (count($links) === 0) {
            return SubscriptionStatus::pending();
        }

        if (count($activeLinks) === 0) {
            return SubscriptionStatus::expired();
        }

        if (count($validLinks) === 0) {
            return SubscriptionStatus::pending();
        }

        return SubscriptionStatus::valid();
    }

    private function findSubscription($identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotDenormalizeSubscriptionStatusException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }
}
