<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\CreateMutation;

use IssetBV\PaymentBundle\CommandBus\CreateMutation\CreateMutationCommand;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\MutationResponseHandler;
use Money\Formatter\DecimalMoneyFormatter;

/**
 * Class TalosCreateMutationHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class CreateMutationHandler
{
    /**
     * @var DecimalMoneyFormatter
     */
    private $moneyFormatter;

    /**
     * @var MutationResponseHandler
     */
    private $mutationResponseHandler;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * TalosCreateMutationHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param MutationResponseHandler $createMutationResponseHandler
     * @param DecimalMoneyFormatter $decimalMoneyFormatter
     */
    public function __construct(ApiGateway $gateway, MutationResponseHandler $createMutationResponseHandler, DecimalMoneyFormatter $decimalMoneyFormatter)
    {
        $this->gateway = $gateway;
        $this->moneyFormatter = $decimalMoneyFormatter;
        $this->mutationResponseHandler = $createMutationResponseHandler;
    }

    /**
     * @param CreateMutationCommand $command
     */
    public function handle(CreateMutationCommand $command)
    {
        $request = new Request($command->getClientData()->getOrElse(ClientData::empty()), [
            new Service('WalletSystem', 'Transfer', [
                new SingleField('SourceWalletKey', $command->getSourceWalletIdentifier()),
                new SingleField('DestinationWalletKey', $command->getDestinationWalletIdentifier()),
                new SingleField('Amount', $this->moneyFormatter->format($command->getAmount())),
                new SingleField('Description', $command->getDescription()->getOrElse('Transaction generated by system')),
            ]),
        ]);

        $request->addResponseHandler($this->mutationResponseHandler);

        $this->gateway->send([$request]);
    }
}
