<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Cli;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Identifier\UuidPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class ExecutePaymentCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    public function __construct(CommandBus $commandBus, PaymentRepository $paymentRepository)
    {
        parent::__construct('talos:payment:execute');
        $this->commandBus = $commandBus;
        $this->paymentRepository = $paymentRepository;
    }

    protected function configure()
    {
        $this->addArgument('payment', InputArgument::REQUIRED, 'the local identifier of the payment you want to execute');
        $this->addArgument('return_url', InputArgument::OPTIONAL, 'the return URL', 'http://requestb.in/17quvd51');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $console = new SymfonyStyle($input, $output);

        $paymentId = UuidPaymentIdentifier::fromString($input->getArgument('payment'));
        
        $console->writeln('Searching/executing ' . $paymentId->asString());
        
        $this->commandBus->handle(new \IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand(
            $paymentId,
            $input->getArgument('return_url')
        ));

        $this->paymentRepository->optionallyFind($paymentId)
            ->filter(function (Payment $payment) {
                return $payment instanceof ExecutablePayment;
            })
            ->flatMap(function (ExecutablePayment $payment) {
                return $payment->getPaymentUrl();
            })
            ->forAll(function ($paymentUrl) use ($console) {
                $console->success($paymentUrl);
            })
            ->getOrCall(function () use ($console) {
                $console->error('Something went wrong');
            });

    }
}
