<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler;

use IssetBV\PaymentBundle\Domain\Repository\WalletTypeRepository;
use IssetBV\TalosBundle\Entity\TalosWalletType;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\Storage\EntityStore;
use RuntimeException;
use function Functional\each;

/**
 * Class WalletTypeResponseHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class WalletTypeResponseHandler implements ResponseHandler
{
    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var WalletTypeRepository
     */
    private $walletTypeRepository;

    /**
     * WalletTypeResponseHandler constructor.
     *
     * @param WalletTypeRepository $walletTypeRepository
     * @param EntityStore $entityStore
     */
    public function __construct(WalletTypeRepository $walletTypeRepository, EntityStore $entityStore)
    {
        $this->entityStore = $entityStore;
        $this->walletTypeRepository = $walletTypeRepository;
    }

    public function handleRequestObject(RequestObject $requestObject)
    {
        $walletType = $this->walletTypeRepository
            ->optionallyFind($requestObject->getIdentifier())
            ->getOrCall(function () use ($requestObject) {
                $walletType = new TalosWalletType(
                    $requestObject->getIdentifier(),
                    $requestObject->getSingleFieldValue('Currency')->getOrThrow(new RuntimeException()),
                    $requestObject->getSingleFieldValue('AllowUnpairedMutations')->getOrThrow(new RuntimeException()),
                    $requestObject->getSingleFieldValue('CreateReversiblePaymentSubWallet')->getOrThrow(new RuntimeException())
                );

                $this->entityStore->persist($walletType);

                return $walletType;
            });

        $requestObject->getSingleFieldValue('Currency')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setCurrency($value);
            });

        $requestObject->getSingleFieldValue('AllowUnpairedMutations')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setAllowUnpairedMutations($value === 'True');
            });

        $requestObject->getSingleFieldValue('CreateReversiblePaymentSubWallet')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setCreateReversiblePaymentSubWallet($value === 'True');
            });

        $requestObject->getSingleFieldValue('Description')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setDescription($value);
            });

        $requestObject->getSingleFieldValue('MinimumAmount')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setMinimumAmount($value);
            });

        $requestObject->getSingleFieldValue('MaximumAmount')
            ->forAll(function (string $value) use ($walletType) {
                $walletType->setMaximumAmount($value);
            });

        $requestObject->getSingleFieldValue('WalletTypeForReversiblePaymentSubWallet')
            ->flatMap(function (string $identifier) {
                return $this->walletTypeRepository->optionallyFind($identifier);
            })
            ->forAll(function (TalosWalletType $existingWalletType) use ($walletType) {
                $walletType->setWalletTypeForReversiblePaymentSubWallet($existingWalletType);
            });
    }

    /**
     * @param Response $response
     */
    public function handleResponse(Response $response)
    {
        each(array_reverse($response->getRequestObjects()), [$this, 'handleRequestObject']);
    }
}
