<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Generator;

use DateTime;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\WalletOwner;
use Ramsey\Uuid\Uuid;

/**
 * Class ReferenceGenerator.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class ReferenceGenerator
{
    const PROJECT_CODE = 'FD';
    const INVOICE_UNIQUE_SIZE = 6;
    const PAYMENT_UNIQUE_SIZE = 6;
    const WALLET_UNIQUE_SIZE = 8;

    /**
     * @param WalletOwner $walletOwner
     *
     * @return string
     */
    public static function generateWalletReference(WalletOwner $walletOwner): string
    {
        return sprintf('%s-%s-%s',
            self::PROJECT_CODE,
            $walletOwner->getId(),
            self::unique(self::WALLET_UNIQUE_SIZE)
        );
    }

    /**
     * @param string $identifier
     *
     * @return string
     */
    public static function generateInvoiceReference(string $identifier): string
    {
        $now = new DateTime();

        return sprintf(
            '%s-%s%s-%d-%s',
            self::PROJECT_CODE,
            $now->format('Ymd'),
            $now->format('His'),
            $identifier,
            self::unique(self::INVOICE_UNIQUE_SIZE)
        );
    }

    /**
     * @param Payment $payment
     *
     * @return string
     */
    public static function generatePaymentReference(Payment $payment)
    {
        $now = new DateTime();

        return sprintf(
            '%s-%s-%s%s-%s',
            self::PROJECT_CODE,
            mb_strtoupper($payment->getPaymentMethod()->getDisplayName()),
            $now->format('Ymd'),
            $now->format('His'),
            self::unique(self::PAYMENT_UNIQUE_SIZE)
        );
    }

    /**
     * @param int $length
     *
     * @return string
     */
    private static function unique(int $length)
    {
        return mb_strtoupper(mb_substr(Uuid::uuid4()->getHex(), 0, min($length, 32)));
    }
}
