<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Mapper;

use DateTime;
use IssetBV\PaymentBundle\Domain\MutableMutation;
use IssetBV\PaymentBundle\Domain\Mutation;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\MutationRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Domain\Repository\WalletRepository;
use IssetBV\PaymentBundle\Domain\Wallet;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Mapper\Exception\MapperException;
use Money\MoneyParser;

/**
 * Class MutationMapper.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class MutationMapper
{
    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var MutationRepository
     */
    private $mutationRepository;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var WalletRepository
     */
    private $walletRepository;

    /**
     * MutationMapper constructor.
     *
     * @param MutationRepository $mutationRepository
     * @param PaymentRepository $paymentRepository
     * @param WalletRepository $walletRepository
     * @param MoneyParser $moneyParser
     */
    public function __construct(MutationRepository $mutationRepository, PaymentRepository $paymentRepository, WalletRepository $walletRepository, MoneyParser $moneyParser)
    {
        $this->mutationRepository = $mutationRepository;
        $this->paymentRepository = $paymentRepository;
        $this->walletRepository = $walletRepository;
        $this->moneyParser = $moneyParser;
    }

    /**
     * @param RequestObject $requestObject
     * @param MutableMutation $mutation
     */
    public function map(RequestObject $requestObject, MutableMutation $mutation)
    {
        $mutation->setAmount($this->moneyParser->parse(
            $requestObject->getSingleFieldValue('Amount')->getOrThrow(MapperException::propertyNotFound('Amount'))
        // @todo currency incorrectly assumes euros should be loaded from wallet
        ));

        $mutation->setDateTime(new DateTime(
            $requestObject->getSingleFieldValue('CreateDateTime')->getOrThrow(MapperException::propertyNotFound('CreateDateTime'))
        ));

        $mutation->setDescription(
            $requestObject->getSingleFieldValue('Description')->getOrThrow(MapperException::propertyNotFound('Description'))
        );

        $requestObject->getSingleFieldValue('PairedTo')
            ->flatMap(function (string $pairedMutationReference) {
                return $this->mutationRepository->findWithRemoteIdentifier($pairedMutationReference);
            })
            ->forAll(function (Mutation $pairedMutation) use ($mutation) {
                $mutation->setPairedMutation($pairedMutation);
            });

        $requestObject->getSingleFieldValue('LinkedPaymentKey')
            ->flatMap(function (string $paymentKey) {
                return $this->paymentRepository->findOneByRemoteIdentifier($paymentKey);
            })
            ->forAll(function (Payment $payment) use ($mutation) {
                $mutation->setPayment($payment);
            })
            ->getOrCall(function () {
                // PAYMENT IS MISSING.. PANIC!
                // Or just chill and mark it as missing, queue it for loading whatever!
            });

        $requestObject->getSingleFieldValue('WalletKey')
            ->flatMap(function (string $walletKey) {
                return $this->walletRepository->findByRemoteIdentifier($walletKey);
            })
            ->forAll(function (Wallet $wallet) use ($mutation) {
                $mutation->setWallet($wallet);
            })
            ->getOrThrow(MapperException::propertyNotFound('WalletKey'));
    }
}
