<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Entity;

use DateInterval;
use DateTime;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\Exception\InvalidPaymentIntervalException;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use PhpOption\Option;

/**
 * @author Tim Fennis <tim@isset.nl>
 * @ORM\Entity()
 * @ORM\Table(name="subscriptions_terms")
 */
class SubscriptionTerm
{
    /**
     * @var int
     * @ORM\Id()
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var Subscription
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Subscription\Subscription")
     */
    private $subscription;

    /**
     * @var Payment
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment")
     */
    private $payment;

    /**
     * @var DateTime
     * @ORM\Column(name="date_time", type="datetime", nullable=false)
     */
    private $dateTime;

    /**
     * @var Interval
     * @ORM\Embedded(class="Interval")
     */
    private $termInterval;

    /**
     * @var bool
     * @ORM\Column(name="activity_override", type="boolean", nullable=false)
     */
    private $override;

    public function __construct(Subscription $subscription, DateTime $startTime, Payment $payment = null, Interval $termInterval = null, $activityOverride = false)
    {
        $this->subscription = $subscription;
        $this->payment = $payment;
        $this->dateTime = $startTime;
        $this->termInterval = $termInterval ?: $subscription->getPaymentInterval();
        $this->override = $activityOverride;
    }

    public function hasSuccessfulPayment(): bool
    {
        return $this->getPayment()
            ->map(function (Payment $payment) {
                return $payment->getRemoteStatus() === 'Success';
            })
            ->getOrElse(false);
    }

    /**
     * Return true if this term has a successful payment or has override enabled.
     *
     * @return bool
     */
    public function isValid(): bool
    {
        return $this->hasSuccessfulPayment() || $this->isOverrideEnabled();
    }

    /**
     * @return DateInterval
     */
    public function getTermInterval(): DateInterval
    {
        return $this->termInterval->getDateInterval()->getOrThrow(new InvalidPaymentIntervalException());
    }

    /**
     * @return DateTime
     */
    public function getDateTime(): DateTime
    {
        return $this->dateTime;
    }

    /**
     * @return Subscription
     */
    public function getSubscription(): Subscription
    {
        return $this->subscription;
    }

    /**
     * @return Option
     */
    public function getPayment(): Option
    {
        return Option::fromValue($this->payment);
    }

    public function isOverrideEnabled(): bool
    {
        return $this->override;
    }

    /**
     * @return bool
     */
    public function isActive(): bool
    {
        $endOfThisTerm = clone $this->dateTime;
        $endOfThisTerm->add($this->getTermInterval());

        return (new DateTime())->getTimestamp() < $endOfThisTerm->getTimestamp();
    }
}
