<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\CreateSubscription;

use DateInterval;
use Doctrine\Common\Persistence\ObjectManager;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Entity\Subscription;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class CreateSubscriptionHandler
{
    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    public function __construct(ObjectManager $objectManager, PaymentRepository $paymentRepository)
    {
        $this->objectManager = $objectManager;
        $this->paymentRepository = $paymentRepository;
    }

    public function handle(CreateSubscriptionCommand $command)
    {
        $subscription = $command->getInitialDateInterval()
            ->map(function (DateInterval $initialInterval) use ($command) {
                return Subscription::createSubscriptionWithInitialInterval(
                    $command->getSubscriptionInterval(),
                    $command->getSubscriptionStartDate(),
                    $this->findPayment($command->getOriginalPaymentIdentifier()),
                    $initialInterval
                );
            })
            ->getOrCall(function () use ($command) {
                return Subscription::createSubscription(
                    $command->getSubscriptionInterval(),
                    $command->getSubscriptionStartDate(),
                    $this->findPayment($command->getOriginalPaymentIdentifier())
                );
            });

        $this->objectManager->persist($subscription);
    }

    private function findPayment(LocalPaymentIdentifier $paymentIdentifier): Payment
    {
        return $this->paymentRepository->optionallyFind($paymentIdentifier)
            ->getOrThrow(CannotCreateSubscriptionException::becauseTheOriginalPaymentCouldNoBeFound($paymentIdentifier));
    }
}
