<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\RenewSubscription;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Entity\Subscription;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use League\Tactician\CommandBus;
use Money\Currency;
use Money\Money;
use function Functional\filter;
use function Functional\first;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionHandler
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(CommandBus $commandBus, InvoiceFactory $invoiceFactory, PaymentFactory $paymentFactory, SubscriptionRepository $subscriptionRepository)
    {
        $this->commandBus = $commandBus;
        $this->invoiceFactory = $invoiceFactory;
        $this->paymentFactory = $paymentFactory;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function handle(RenewSubscriptionCommand $command)
    {
        $subscription = $this->findSubscription($command->getSubscriptionIdentifier());

        $renewablePayment = first(filter($subscription->getPayments(), [$this, 'isExecutablePayment']), [$this, 'paymentTypeIsFirst']);

        $newPayment = $this->paymentFactory->createRecurringPayment(
            $this->invoiceFactory->createInvoice(),
            new Money(299, new Currency('EUR')),//@todo do not hard code subscription amount
            $renewablePayment
        );

        $this->commandBus->handle(new ExecutePaymentCommand($newPayment->getId()));

        $subscription->createNewTerm($newPayment);
    }

    public function isExecutablePayment($payment)
    {
        return $payment instanceof ExecutablePayment;
    }

    public function paymentTypeIsFirst(ExecutablePayment $payment)
    {
        return $payment->getType()->isFirst();
    }

    private function findSubscription(SubscriptionIdentifier $identifier): Subscription
    {
        return $this->subscriptionRepository->optionallyFind($identifier)
            ->getOrThrow(CannotRenewSubscriptionException::becauseTheSubscriptionCouldNotBeFound($identifier));
    }
}
