<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use Doctrine\Common\Collections\Criteria;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\RenewSubscriptionCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use IssetBV\PaymentBundle\Entity\Subscription;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use function Functional\each;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionsCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository
    ) {
        parent::__construct('subscription:renewAll');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function configure()
    {
        // do nothing for now
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->updateSubscriptions($input, $output);
        $this->renewSubscriptions($input, $output);
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    private function updateSubscriptions(InputInterface $input, OutputInterface $output)
    {
        $console = new SymfonyStyle($input, $output);

        $subscriptions = $this->subscriptionRepository->matching(Criteria::create());

        $console->title(sprintf('Updating %s subscriptions', $subscriptions->count()));

        try {
            $console->progressStart($subscriptions->count());

            each($subscriptions, function (Subscription $subscription) use ($console) {
                $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($subscription->getId()));
                $console->progressAdvance(1);
            });

            $console->progressFinish();
            $console->success('Subscriptions updated');
        } catch (\Exception $e) {
            $console->progressFinish();
            $console->caution('Exception: ' . $e->getMessage());
        }
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     */
    private function renewSubscriptions(InputInterface $input, OutputInterface $output)
    {
        $console = new SymfonyStyle($input, $output);

        $criteria = Criteria::create();
        $criteria->where(Criteria::expr()->eq('denormalizedStatus', SubscriptionStatus::expired()->getCode()));

        $subscriptions = $this->subscriptionRepository->matching($criteria);

        $console->title(sprintf('Renewing %s subscriptions', $subscriptions->count()));

        try {
            $console->progressStart($subscriptions->count());

            each($subscriptions, function (Subscription $subscription) use ($console) {
                $this->commandBus->handle(new RenewSubscriptionCommand($subscription->getId()));
                $console->progressAdvance(1);
            });

            $console->progressFinish();
            $console->success('Subscriptions updated');
        } catch (\Exception $e) {
            $console->progressFinish();
            $console->caution('Exception: ' . $e->getMessage());

            throw $e;
        }
    }
}
