<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Response;

use DateTime;
use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\JsonArray;
use IssetBV\Json\JsonObject;
use IssetBV\Json\JsonThing;
use IssetBV\TalosBundle\Gateway\Request\Service;
use PhpOption\Option;

/**
 * Class Response.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class WebResponse extends Response
{
    /**
     * @var string
     */
    private $key;

    /**
     * @var Service[]
     */
    private $services;

    /**
     * @var Status
     */
    private $status;

    /**
     * Response constructor.
     *
     * @param string $key
     * @param string $nonce
     * @param Status $status
     * @param DateTime $time
     * @param string $userCulture
     * @param Service[] $services
     * @param RequestObject[] $requestObjects
     */
    protected function __construct(string $key, string $nonce, Status $status, DateTime $time, string $userCulture, array $services, array $requestObjects)
    {
        parent::__construct($nonce, $time, $userCulture, $requestObjects);
        $this->key = $key;
        $this->status = $status;
        $this->services = $services;
    }

    /**
     * @return Option
     */
    public function getKey(): Option
    {
        return Option::ensure($this->key);
    }

    /**
     * @return Status
     */
    public function getStatus(): Status
    {
        return $this->status;
    }

    /**
     * @return array|Service[]
     */
    public function getServices(): array
    {
        return $this->services;
    }

    /**
     * @param JsonThing|JsonObject[]|JsonArray[] $jsonThing
     *
     * @throws DecodeException
     *
     * @return Response
     */
    public static function fromJsonThing(JsonThing $jsonThing)
    {
        $requestObjects = [];

        if ($jsonThing->hasProperty('RequestObjects')) {
            $requestObjects = $jsonThing['RequestObjects']->map(
                function (JsonThing $jsonThing): RequestObject {
                    return RequestObject::fromJsonThing($jsonThing);
                }
            );
        }

        return new self(
            $jsonThing->hasProperty('Key') ? $jsonThing['Key'] : '',
            (string) $jsonThing['Nonce'],
            Status::fromJsonThing($jsonThing['Status']),
            new DateTime($jsonThing['Time']),
            (string) $jsonThing['UserCulture'],
            $jsonThing['Services']->map(
                function (JsonThing $jsonThing): Service {
                    return Service::fromJsonThing($jsonThing);
                }
            ),
            $requestObjects
        );
    }
}
