<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\SynchronizeIssuers;

use Doctrine\Common\Collections\Criteria;
use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\JsonThing;
use IssetBV\PaymentBundle\CommandBus\SynchronizeIssuers\SynchronizeIssuersCommand;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentMethodRepository;
use IssetBV\TalosBundle\Entity\TalosPaymentIssuer;
use IssetBV\TalosBundle\Exception\CannotSynchronizeIssuersException;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Storage\EntityStore;
use PhpOption\Option;
use function Functional\each;
use function Functional\first;

/**
 * Class SynchronizeIssuersHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizeIssuersHandler
{
    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentIssuerRepository
     */
    private $paymentIssuerRepository;

    /**
     * @var PaymentMethodRepository
     */
    private $paymentMethodRepository;

    /**
     * SynchronizeIssuersHandler constructor.
     *
     * @param EntityStore $entityStore
     * @param ApiGateway $gateway
     * @param PaymentIssuerRepository $paymentIssuerRepository
     * @param PaymentMethodRepository $paymentMethodRepository
     */
    public function __construct(EntityStore $entityStore, ApiGateway $gateway, PaymentIssuerRepository $paymentIssuerRepository, PaymentMethodRepository $paymentMethodRepository)
    {
        $this->entityStore = $entityStore;
        $this->gateway = $gateway;
        $this->paymentIssuerRepository = $paymentIssuerRepository;
        $this->paymentMethodRepository = $paymentMethodRepository;
    }

    /**
     * @param SynchronizeIssuersCommand $command
     *
     * @throws CannotSynchronizeIssuersException
     */
    public function handle(SynchronizeIssuersCommand $command)
    {
        try {
            $issuerMapping = [];
            foreach ($this->paymentIssuerRepository->matching(Criteria::create()) as $paymentIssuer) {
                $issuerMapping[$paymentIssuer->getCode()] = $paymentIssuer;
            }

            each($issuerMapping, function (TalosPaymentIssuer $paymentIssuer) {
                $paymentIssuer->setSeenInLastUpdate(false);
            });

            /** @var JsonThing[] $remoteIssuers */
            $remoteIssuers = $this->fetchIssuerData($command->getPaymentMethodServiceName())
                ->getOrThrow(CannotSynchronizeIssuersException::becauseTheIssuersCouldNotBeLoaded());

            foreach ($remoteIssuers as $remoteIssuer) {
                $issuerIdentifier = $remoteIssuer['Value'];

                if (false === array_key_exists($issuerIdentifier, $issuerMapping)) {

                    $paymentMethod = $this->paymentMethodRepository
                        ->findByServiceName($command->getPaymentMethodServiceName())
                        ->getOrThrow(CannotSynchronizeIssuersException::becauseThePaymentMethodCouldNotBeFound($command->getPaymentMethodServiceName()));

                    $issuer = new TalosPaymentIssuer(
                        $issuerIdentifier,
                        $remoteIssuer['Name'],
                        $remoteIssuer['GroupName'],
                        $remoteIssuer['Default'],
                        $paymentMethod
                    );

                    $issuerMapping[$issuerIdentifier] = $issuer;
                    $this->entityStore->persist($issuer);
                } else {
                    $issuerMapping[$issuerIdentifier]->update($remoteIssuer['Name'], $remoteIssuer['GroupName'], $remoteIssuer['Default']);
                }
            }
        } catch (DecodeException $e) {
            throw new CannotSynchronizeIssuersException('Unable to decode JSON data', 0, $e);
        }
    }

    /**
     * @param string $serviceName
     *
     * @return Option<JsonThing[]>
     */
    private function fetchIssuerData(string $serviceName): Option
    {
        // Loads the service spec for the given payment method
        // We only actually know for sure that the response format works with ideal other payment methods will probalby
        // throw errors if loaded through this method
        $data = $this->gateway->loadServiceSpecification(new Service($serviceName, 'Pay'));


        return Option::ensure(first(
            $data['BulkRequestSpecification']['Request']['Services'],
            function (JsonThing $service) {
                return strcasecmp($service['Name'], 'ideal') === 0;
            }
        ))->flatMap(function (JsonThing $service) {
            return $this->getIssuersFromService($service);
        });
    }

    /**
     * @param JsonThing $service
     *
     * @throws DecodeException
     *
     * @return Option
     */
    private function getIssuersFromService(JsonThing $service): Option
    {
        return Option::ensure(first(
            $service['Actions'],
            function (JsonThing $action) {
                return strcasecmp($action['Name'], 'pay') === 0;
            }
        ))->flatMap(function (JsonThing $action) {
            return $this->getIssuersFromAction($action);
        });
    }

    /**
     * @param JsonThing $action
     *
     * @throws DecodeException
     *
     * @return Option
     */
    private function getIssuersFromAction(JsonThing $action): Option
    {
        return Option::ensure(first(
            $action['InputFields'],
            function (JsonThing $inputField) {
                return strcasecmp($inputField['Name'], 'issuer') === 0;
            }
        ))->map(function (JsonThing $inputField) {
            return $inputField['ListItems'];
        });
    }
}
