<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Response;

use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\JsonThing;
use IssetBV\TalosBundle\Gateway\Shared\Field;
use IssetBV\TalosBundle\Gateway\Shared\FieldContainer;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use PhpOption\Option;

/**
 * Class RequestObject.
 *
 * I literally can't tell you why someone decided to call this a request object. It's only really part of responses though.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class RequestObject
{
    use FieldContainer;

    /**
     * @var string
     */
    private $owner;

    /**
     * @var string
     */
    private $type;

    /**
     * @var string
     */
    private $identifier;

    /**
     * @var string
     */
    private $reference;

    /**
     * RequestObject constructor.
     *
     * @param string $owner
     * @param string $type
     * @param string $identifier
     * @param SingleField[] $fields
     * @param string $reference
     */
    private function __construct(string $owner, string $type, string $identifier, string $reference, array $fields)
    {
        $this->owner = $owner;
        $this->type = $type;
        $this->identifier = $identifier;
        $this->reference = $reference;
        $this->setFields($fields);
    }

    /**
     * @return string
     */
    public function getOwner(): string
    {
        return $this->owner;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return string
     */
    public function getIdentifier(): string
    {
        return $this->identifier;
    }

    /**
     * @return Option containing a string
     */
    public function getReference(): Option
    {
        return Option::ensure($this->reference);
    }

    /**
     * @param JsonThing $jsonThing
     *
     * @throws DecodeException
     *
     * @return RequestObject
     */
    public static function fromJsonThing(JsonThing $jsonThing)
    {
        return new self(
            $jsonThing['ObjectOwner'],
            $jsonThing['ObjectType'],
            $jsonThing['ObjectIdentifier'],
            $jsonThing->hasProperty('ObjectReference') ? $jsonThing['ObjectReference'] : '',
            $jsonThing['Fields']->map(
                function (JsonThing $jsonThing): Field {
                    return Field::fromJsonThing($jsonThing);
                }
            )
        );
    }
}
