<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Entity;

use DateTime;
use Doctrine\ORM\Mapping as ORM;
use IssetBV\PaymentBundle\Domain\MutableMutation;
use IssetBV\PaymentBundle\Domain\Mutation;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Wallet;
use Money\Currency;
use Money\Money;
use PhpOption\Option;

/**
 * Class TalosMutation.
 *
 * @author Tim Fennis <tim@isset.nl>
 * @ORM\Entity(repositoryClass="IssetBV\TalosBundle\Repository\DoctrineMutationRepository")
 * @ORM\Table(name="mutations")
 */
class TalosMutation implements MutableMutation
{
    /**
     * @var string
     * @ORM\Id()
     * @ORM\Column(name="id", type="string")
     */
    private $id;

    /**
     * @var Money
     * @ORM\Column(name="amount", type="money", nullable=false)
     */
    private $amount;

    /**
     * @var DateTime
     * @ORM\Column(name="date_time", type="datetime", nullable=true)
     */
    private $dateTime;

    /**
     * @var string
     * @ORM\Column(name="description", type="string", nullable=false)
     */
    private $description;

    /**
     * @var Mutation
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Mutation")
     */
    private $pairedMutation;

    /**
     * @var Payment
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Payment", inversedBy="mutations")
     */
    private $payment;

    /**
     * @var Wallet
     * @ORM\ManyToOne(targetEntity="IssetBV\PaymentBundle\Domain\Wallet", inversedBy="mutations")
     */
    private $wallet;

    /**
     * TalosMutation constructor.
     *
     * @param string $id
     */
    public function __construct(string $id)
    {
        $this->id = $id;
        $this->amount = new Money(0, new Currency('EUR'));
        $this->description = '';
    }

    /**
     * @return string
     */
    public function getId(): string
    {
        return $this->id;
    }

    /**
     * This alias is part of the Mutation interface.
     *
     * @see Mutation
     *
     * @return string
     */
    public function getIdentifier(): string
    {
        return $this->getId();
    }

    /**
     * @return DateTime
     */
    public function getDateTime()
    {
        return $this->dateTime;
    }

    /**
     * @return Option
     */
    public function getPairedMutation(): Option
    {
        return Option::ensure($this->pairedMutation);
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @return Option
     */
    public function getWallet(): Option
    {
        return Option::ensure($this->wallet);
    }

    /**
     * @return Money
     */
    public function getAmount(): Money
    {
        return $this->amount;
    }

    /**
     * @return Option
     */
    public function getPayment(): Option
    {
        return Option::ensure($this->payment);
    }

    /**
     * @param DateTime $dateTime
     */
    public function setDateTime(DateTime $dateTime)
    {
        $this->dateTime = $dateTime;
    }

    /**
     * @param Mutation $mutation
     */
    public function setPairedMutation(Mutation $mutation)
    {
        $this->pairedMutation = $mutation;
    }

    /**
     * @param string $description
     */
    public function setDescription(string $description)
    {
        $this->description = $description;
    }

    /**
     * @param Wallet $wallet
     */
    public function setWallet(Wallet $wallet)
    {
        $this->wallet = $wallet;
    }

    /**
     * @param Money $amount
     */
    public function setAmount(Money $amount)
    {
        $this->amount = $amount;
    }

    /**
     * @param Payment $payment
     */
    public function setPayment(Payment $payment)
    {
        $this->payment = $payment;
    }
}
