<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\ResponseHandler;

use IssetBV\PaymentBundle\CommandBus\SynchronizeWallet\CannotSynchronizeWalletException;
use IssetBV\PaymentBundle\Domain\Repository\AutoTopupProfileRepository;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\PaymentBundle\Domain\Repository\WalletRepository;
use IssetBV\PaymentBundle\Domain\Repository\WalletTypeRepository;
use IssetBV\TalosBundle\Entity\TalosWallet;
use IssetBV\TalosBundle\Entity\TalosWalletType;
use IssetBV\TalosBundle\Gateway\Exception\WalletResponseException;
use IssetBV\TalosBundle\Gateway\Response\Handler\ResponseHandler;
use IssetBV\TalosBundle\Gateway\Response\RequestObject;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Storage\EntityStore;

/**
 * Class SynchronizeWalletResponseHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class WalletResponseHandler implements ResponseHandler
{
    /**
     * @var AutoTopupProfileRepository
     */
    private $autoTopupProfileRepository;

    /**
     * @var EntityStore
     */
    private $entityStore;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var WalletRepository
     */
    private $walletRepository;

    /**
     * @var WalletTypeRepository
     */
    private $walletTypeRepository;

    /**
     * WalletResponseHandler constructor.
     *
     * @param AutoTopupProfileRepository $autoTopupProfileRepository
     * @param EntityStore $entityStore
     * @param PaymentRepository $paymentRepository
     * @param WalletRepository $walletRepository
     * @param WalletTypeRepository $walletTypeRepository
     */
    public function __construct(
        AutoTopupProfileRepository $autoTopupProfileRepository,
        EntityStore $entityStore,
        PaymentRepository $paymentRepository,
        WalletRepository $walletRepository,
        WalletTypeRepository $walletTypeRepository
    ) {
        $this->autoTopupProfileRepository = $autoTopupProfileRepository;
        $this->entityStore = $entityStore;
        $this->paymentRepository = $paymentRepository;
        $this->walletRepository = $walletRepository;
        $this->walletTypeRepository = $walletTypeRepository;
    }

    /**
     * @param Response $response
     */
    public function handleResponse(Response $response)
    {
        $this->handleRequestObject($response->getOneRequestObject()->getOrThrow(new WalletResponseException('Cannot handle wallet response because __CLASS__ has a problem handling multiple or they werent found')));
    }

    private function handleRequestObject(RequestObject $requestObject)
    {
        $wallet = $this->findOrCreateWallet($requestObject);

        $wallet->setReference($requestObject->getReference()->getOrThrow(new WalletResponseException('Wallet response is invalid because no reference could be found')));

        $requestObject->getSingleFieldValue('TotalBalance')
            ->forAll(function (string $totalBalance) use ($wallet) {
                $wallet->setBalance($totalBalance);
            });

        $requestObject->getSingleFieldValue('LocalBalance')
            ->forAll(function (string $localBalance) use ($wallet) {
                $wallet->setLocalBalance($localBalance);
            });

        $requestObject->getField('Status')
            ->flatMap(
                function (GroupField $groupField) {
                    return $groupField->getSingleFieldValue('StatusCode');
                }
            )
            ->forAll(
                function (string $statusCode) use ($wallet) {
                    $wallet->setRemoteStatus($statusCode);
                }
            );

        $requestObject->getSingleFieldValue('TypeKey')
            ->flatMap(function (string $typeKey) {
                $this->walletTypeRepository->optionallyFind($typeKey);
            })
            ->forAll(function (TalosWalletType $walletType) use ($wallet) {
                $wallet->setType($walletType);
            });

        $autoTopupProfile = $requestObject->getField('AutoTopupLinkInfo')
            ->flatMap(function (GroupField $linkInfo) {
                return $linkInfo->getSingleFieldValue('AutoTopupProfileKey');
            })
            ->map(function (string $autoTopupProfileKey) {
                return $this->autoTopupProfileRepository->optionallyFind($autoTopupProfileKey)
                    ->getOrThrow(new CannotSynchronizeWalletException("Cannot synchronize wallet exception because the auto topup profile with key $autoTopupProfileKey  is not known"));
            })
            ->getOrElse(null);

        $wallet->setAutoTopupProfile($autoTopupProfile);

        $autoTopupPayment = $requestObject->getField('AutoTopupLinkInfo')
            ->flatMap(function (GroupField $linkInfo) {
                return $linkInfo->getSingleFieldValue('PaymentKey');
            })
            ->flatMap(function (string $paymentKey) {
                //@todo if the payment could not be found we should request the #missing_information
                return $this->paymentRepository->findOneByRemoteIdentifier($paymentKey);
            })
            ->getOrElse(null);

        $wallet->setAutoTopupPayment($autoTopupPayment);
    }

    /**
     * Tries to find an existing wallet based on the request object otherwise it will create a new one.
     *
     * @param RequestObject $requestObject
     *
     * @return TalosWallet
     */
    private function findOrCreateWallet(RequestObject $requestObject): TalosWallet
    {
        return $this->walletRepository->findByRemoteIdentifier($requestObject->getIdentifier())
            ->getOrCall(function () use ($requestObject) {
                $this->createWallet($requestObject);
            });
    }

    /**
     * Creates a new Wallet based on the request object.
     *
     * @param RequestObject $requestObject
     *
     * @return TalosWallet
     */
    private function createWallet(RequestObject $requestObject): TalosWallet
    {
        $walletType = $requestObject
            ->getSingleFieldValue('TypeKey')
            ->flatMap(function (string $typeKey) {
                return $this->walletTypeRepository->optionallyFind($typeKey);
            })
            ->getOrThrow(new WalletResponseException());

        $wallet = new TalosWallet(
            $walletType,
            $requestObject->getIdentifier(),
            $requestObject->getReference()->getOrThrow(new WalletResponseException('Required property Reference not found in requestObject')) // @todo will this fuck up when we're syncing subwallets?
        );

        $this->entityStore->persist($wallet);

        //@todo removed the flush from here, add persistNow to the EntityStore interface if it's really needed

        return $wallet;
    }
}
