<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\SynchronizePayment;

use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\CannotSynchronizePaymentException;
use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizeLocalPaymentCommand;
use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizeRemotePaymentCommand;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Exception\ResponseException;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\Mapper\Exception\MapperException;
use IssetBV\TalosBundle\ResponseHandler\Factory\PaymentResponseHandlerFactory;

/**
 * Class TalosSynchronizePaymentHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizePaymentHandler
{
    const ACTION_NAME = 'Status';

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var PaymentResponseHandlerFactory
     */
    private $paymentResponseHandlerFactory;

    /**
     * TalosSynchronizePaymentHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param PaymentResponseHandlerFactory $paymentResponseHandlerFactory
     * @param PaymentRepository $paymentRepository
     */
    public function __construct(
        ApiGateway $gateway,
        PaymentResponseHandlerFactory $paymentResponseHandlerFactory,
        PaymentRepository $paymentRepository
    ) {
        $this->gateway = $gateway;
        $this->paymentResponseHandlerFactory = $paymentResponseHandlerFactory;
        $this->paymentRepository = $paymentRepository;
    }

    /**
     * @param SynchronizeLocalPaymentCommand $command
     *
     * @throws CannotSynchronizePaymentException
     */
    public function handleLocal(SynchronizeLocalPaymentCommand $command)
    {
        // We load the payment from the DB to ensure it exists. Better to fail early
        $this->findPayment($command->getPaymentIdentifier()); // throws

        $request = new Request(ClientData::empty(), [
            new Service('General', self::ACTION_NAME, [
                new SingleField('ObjectType', 'Payment'),
                new SingleField('ObjectReference', (string) $command->getPaymentIdentifier()),
            ]),
        ]);

        //@todo this is possibly the incorrect type of handler
        $this->executeRequest($request);
    }

    /**
     * @param SynchronizeLocalPaymentCommand $command
     *
     * @throws CannotSynchronizePaymentException
     */
    public function handleRemote(SynchronizeRemotePaymentCommand $command)
    {
        $request = new Request(ClientData::empty(), [
            new Service('General', self::ACTION_NAME, [
                new SingleField('ObjectType', 'Payment'),
                new SingleField('ObjectIdentifier', $command->getPaymentIdentifier()), //@todo this is the talos ID that we don't need to use AFAIK
            ]),
        ]);

        $this->executeRequest($request);
    }

    protected function findPayment(LocalPaymentIdentifier $paymentIdentifier): Payment
    {
        return $this->paymentRepository->optionallyFind($paymentIdentifier)
            ->getOrThrow(CannotSynchronizePaymentException::becauseThePaymentCouldNotBeFound($paymentIdentifier));
    }

    /**
     * @param Request $request
     *
     * @throws ResponseException
     */
    private function executeRequest($request)
    {
        //@todo this is possibly the incorrect type of handler
        $request->addResponseHandler($this->paymentResponseHandlerFactory->createResponseHandler());

        try {
            $this->gateway->send([$request]);
        } catch (MapperException $e) {
//            throw CannotSynchronizePaymentException::becauseTheGatewayResponseCouldNotBeMapped($e);
        }
    }
}
