<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Cli;

use IssetBV\PaymentBundle\CommandBus\SynchronizePayment\SynchronizeLocalPaymentCommand;
use IssetBV\PaymentBundle\Domain\Identifier\UuidPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Cli\Renderer\PaymentRenderer;
use League\Tactician\CommandBus;
use PhpOption\LazyOption;
use Ramsey\Uuid\Uuid;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * This command accepts 1 argument with the database id of the payment.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class SynchronizePaymentCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var PaymentRenderer
     */
    private $paymentRenderer;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * SynchronizePaymentCommand constructor.
     *
     * @param CommandBus $commandBus
     * @param PaymentRepository $paymentRepository
     * @param PaymentRenderer $paymentRenderer
     */
    public function __construct(CommandBus $commandBus, PaymentRepository $paymentRepository, PaymentRenderer $paymentRenderer)
    {
        parent::__construct('talos:payment:sync');

        $this->commandBus = $commandBus;
        $this->paymentRepository = $paymentRepository;
        $this->paymentRenderer = $paymentRenderer;
    }

    protected function configure()
    {
        $this->addArgument('payment', InputArgument::REQUIRED, 'The db ID of the payment');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $paymentIdentifier = $input->getArgument('payment');

        $payment = $this->paymentRepository->findOneByRemoteIdentifier($paymentIdentifier)
            ->orElse(new LazyOption(function () use ($paymentIdentifier) {
                return $this->paymentRepository->findOneByReference($paymentIdentifier);
            }))
            ->orElse(new LazyOption(function () use ($paymentIdentifier) {
                return $this->paymentRepository->optionallyFind(new UuidPaymentIdentifier(Uuid::fromString($paymentIdentifier)));
            }))
            ->getOrThrow(new InvalidArgumentException('Payment not found: ' . $paymentIdentifier));

        $this->commandBus->handle(SynchronizeLocalPaymentCommand::fromPayment($payment));

        $this->paymentRenderer->render(
            $this->paymentRepository->optionallyFind($payment->getId())->getOrThrow(new RuntimeException()),
            $output
        );
    }
}
