<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\CommandBus\ExecutePayment;

use Exception;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Payment\PaymentType;
use IssetBV\PaymentBundle\Domain\PaymentMethod;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
final class CannotExecutePaymentException extends Exception
{
    public static function becauseNoIssuerWasSet($paymentIdentifier = 'unknown'): CannotExecutePaymentException
    {
        return new self("Unable to create service request for ideal payment (id: $paymentIdentifier) because no payment issuer (bank) was set in the payment");
    }

    public static function becauseTheReturnUrlWasNotSet($paymentIdentifier = 'unknown'): CannotExecutePaymentException
    {
        return new self("Unable to execute payment (id: $paymentIdentifier) because the return URL was not set in the payment");
    }

    public static function becauseThePaymentDoesNotExist(LocalPaymentIdentifier $paymentId): CannotExecutePaymentException
    {
        return new self('Unable to execute payment because no payment with id ' . $paymentId->asString() . ' could be found');
    }

    public static function becauseThePaymentTypeIsNotSupported(PaymentType $type): CannotExecutePaymentException
    {
        return new self('Unable to execute payment because the payment type \'' . (string) $type . '\' is not supported');
    }

    public static function becauseNoOriginalPaymentIsSet(): CannotExecutePaymentException
    {
        return new self('Unable to execute payment because no original payment is set or no remote identifier exists for original payment');
    }

    public static function becauseOfAResponseException(\Throwable $e)
    {
        return new self('Unable to execute payment because the response received from the gateway while executing the payment was invalid', 0, $e);
    }

    public static function becauseThePaymentMethodIsNotActive(PaymentMethod $paymentMethod)
    {
        return new self(sprintf('The payment method %s is not active until %s', $paymentMethod->getDisplayName(), $paymentMethod->getDateActive()->format('c')));
    }
}
