<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\CancelPayment;

use IssetBV\PaymentBundle\CommandBus\CancelPayment\CancelPaymentCommand;
use IssetBV\PaymentBundle\CommandBus\CancelPayment\CannotCancelPaymentException;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Response\Handler\ThrowingErrorHandler;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class CancelPaymentHandler
{
    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    public function __construct(ApiGateway $gateway, PaymentRepository $paymentRepository)
    {
        $this->gateway = $gateway;
        $this->paymentRepository = $paymentRepository;
    }

    public function handle(CancelPaymentCommand $command)
    {
        $payment = $this->findPayment($command->getPaymentIdentifier());

        $service = new Service('iDEAL', 'Cancel', [
            new SingleField('ObjectType', 'Payment'),
            new SingleField('ObjectIdentifier', $payment->getRemoteIdentifier()
                ->getOrThrow(CannotCancelPaymentException::becauseThePaymentDoesNotHaveARemoteIdentifier($command->getPaymentIdentifier()))),
        ]);

        $request = new Request($command->getClientData()->getOrElse(ClientData::empty()), [$service]);
        //        $request->addResponseHandler();
        $request->addRequestErrorHandler(new ThrowingErrorHandler());

        $this->gateway->send([$request]);
    }

    private function findPayment(LocalPaymentIdentifier $paymentId): ExecutablePayment
    {
        return $this->paymentRepository
            ->optionallyFind($paymentId)
            ->getOrThrow(CannotCancelPaymentException::becauseThePaymentDoesNotExist($paymentId));
    }
}
