<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway\Response;

use DateTime;
use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\JsonThing;
use IssetBV\TalosBundle\Gateway\Request\Service;
use PhpOption\None;
use PhpOption\Option;

/**
 * Class Response.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class Response
{
    /**
     * @var string
     */
    private $nonce;

    /**
     * @var RequestObject[]
     */
    private $requestObjects;

    /**
     * @var DateTime
     */
    private $time;

    /**
     * @var string
     */
    private $userCulture;

    /**
     * Response constructor.
     *
     * @param string $nonce
     * @param DateTime $time
     * @param string $userCulture
     * @param RequestObject[] $requestObjects
     */
    protected function __construct(string $nonce, DateTime $time, string $userCulture, array $requestObjects)
    {
        $this->nonce = $nonce;
        $this->time = $time;
        $this->userCulture = $userCulture;
        $this->requestObjects = $requestObjects;
    }

    /**
     * @return string
     */
    public function getNonce(): string
    {
        return $this->nonce;
    }

    /**
     * Returns an Option for the first RequestObject in this response. This is useful in situations where you know
     * only one request object will be provided.
     *
     * @return Option
     */
    public function getOneRequestObject(): Option
    {
        $requestObject = reset($this->requestObjects) ?: null;

        return Option::ensure($requestObject);
    }

    /**
     * @todo handle the case where multiple request objects of the same type are returned.
     *
     * @param string $searchType
     *
     * @return Option Containing {@see RequestObject}
     */
    public function getOneRequestObjectWithType(string $searchType)
    {
        foreach ($this->requestObjects as $requestObject) {
            if ($requestObject->getType() === $searchType) {
                return Option::ensure($requestObject);
            }
        }

        return None::create();
    }

    /**
     * @param string $typeName
     *
     * @return array
     */
    public function getRequestObjectsWithType($typeName)
    {
        return \Functional\select($this->requestObjects, function (RequestObject $requestObject) use ($typeName) {
            return $requestObject->getType() === $typeName;
        });
    }

    /**
     * @return DateTime
     */
    public function getTime(): DateTime
    {
        return $this->time;
    }

    /**
     * @return string
     */
    public function getUserCulture(): string
    {
        return $this->userCulture;
    }

    /**
     * @return RequestObject[]
     */
    public function getRequestObjects(): array
    {
        return $this->requestObjects;
    }

    /**
     * @return Service[]
     */
    public function getServices(): array
    {
        return [];
    }

    /**
     * @param JsonThing $jsonThing
     *
     * @throws DecodeException
     *
     * @return Response
     */
    public static function fromJsonThing(JsonThing $jsonThing)
    {
        $requestObjects = [];

        if ($jsonThing->hasProperty('RequestObjects')) {
            $requestObjects = $jsonThing['RequestObjects']->map(
                function (JsonThing $jsonThing): RequestObject {
                    return RequestObject::fromJsonThing($jsonThing);
                }
            );
        }

        return new self(
            (string) $jsonThing['Nonce'],
            new DateTime($jsonThing['Time']),
            (string) $jsonThing['UserCulture'],
            $requestObjects
        );
    }
}
