<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Gateway;

use DateTime;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Psr7\Request;
use IssetBV\Json\Exception\DecodeException;
use IssetBV\Json\Exception\InvalidPropertyAccessException;
use IssetBV\Json\JsonThing;
use IssetBV\Json\SimpleDecoder;
use IssetBV\TalosBundle\Exception\ApiGatewayException;
use IssetBV\TalosBundle\Exception\OpenSSLException;
use IssetBV\TalosBundle\Gateway\Config\GatewayConfig;
use IssetBV\TalosBundle\Gateway\Logger\GatewayLogger;
use IssetBV\TalosBundle\Gateway\Logger\LogData;
use IssetBV\TalosBundle\Gateway\Request\Batch;
use IssetBV\TalosBundle\Gateway\Request\Request as TalosApiRequest;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Response\ApiResponseWrapper;
use IssetBV\TalosBundle\Gateway\Response\Response;
use IssetBV\TalosBundle\ResponseHandler\DumpingResponseHandler;
use PhpOption\Option;
use Psr\Log\LoggerInterface;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;

/**
 * Class ApiGateway.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
final class ApiGateway
{
    /**
     * @var GatewayConfig
     */
    private $gatewayConfig;

    /**
     * @var Option<GatewayLogger>
     */
    private $gatewayLogger;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param GatewayConfig $gatewayConfig
     * @param GatewayLogger $gatewayLogger
     */
    public function __construct(GatewayConfig $gatewayConfig, GatewayLogger $gatewayLogger = null)
    {
        $this->gatewayConfig = $gatewayConfig;
        $this->gatewayLogger = Option::ensure($gatewayLogger);
    }

    /**
     * @param Service $service
     *
     * @return JsonThing
     */
    public function loadServiceSpecification(Service $service)
    {
        $request = [
            'SenderCode' => $this->gatewayConfig->getSenderCode(),
            'Services' => [
                $service->toSerializable(),
            ],
        ];

        return $this->sendData(json_encode($request), 'json/?op=specification', Uuid::uuid4());
    }

    /**
     * @param TalosApiRequest[] $requests
     * @param string $channel
     */
    public function send(array $requests, string $channel = 'BatchFile')
    {
        $httpRequestId = Uuid::uuid4();
        $batch = new Batch($requests, $channel);

        $apiResponse = $this->executeBatch($batch, $httpRequestId);
        $wrapper = new ApiResponseWrapper($apiResponse);

        /** @var TalosApiRequest $request */
        foreach ($requests as $request) {
            // Show debugging info for failed requests in debug mode
            if ($this->gatewayConfig->isDebugMode()) {
                $request->addRequestErrorHandler(new DumpingResponseHandler());
            }

            try {
                $wrapper->getResponseFor($request)->forAll(
                    function (Response $response) use ($request) {
                        $request->callResponseHandlers($response);
                    }
                );
            } catch (DecodeException $e) {
                $request->callRequestErrorHandlers([
                    'exception' => $e,
                    'response' => $apiResponse,
                ]);
            } catch (InvalidPropertyAccessException $e) {
                $request->callRequestErrorHandlers([
                    'exception' => $e,
                    'response' => $apiResponse,
                ]);
            }
        }
    }

    /**
     * @return Client
     */
    private function getClient(): Client
    {
        static $client;
        if ($client === null) {
            $client = new Client([
                'base_uri' => $this->gatewayConfig->getApiUrl(),
                'timeout' => 20.0,
            ]);
        }

        return $client;
    }

    /**
     * @param Batch $batch
     * @param UuidInterface $requestId
     *
     * @throws ApiGatewayException
     * @throws OpenSSLException
     *
     * @return JsonThing
     */
    private function executeBatch(Batch $batch, UuidInterface $requestId): JsonThing
    {
        return $this->sendData(json_encode($batch, 0), 'json/', $requestId);
    }

    /**
     * @param string $requestData
     * @param string $url
     * @param UuidInterface $requestId
     *
     * @throws ApiGatewayException
     * @throws OpenSSLException
     *
     * @return JsonThing
     */
    private function sendData(string $requestData, string $url, UuidInterface $requestId): JsonThing
    {
        static $sequenceId = 0;
        static $groupId = null;
        $groupId = $groupId ?: Uuid::uuid4();

        $request = new Request('POST', $url, [
            'PSP-Certificate-Signature' => base64_encode($this->gatewayConfig->getPrivateKey()->signData($requestData)),
            'PSP-Certificate-Thumbprint' => $this->gatewayConfig->getPublicKey()->getFingerprint(),
        ], $requestData);

        $requestDate = new DateTime();

        try {
            $response = $this->getClient()->send($request);
        } catch (ClientException $e) {
            $response = $e->getResponse();

            if ($response === null) {
                throw new ApiGatewayException('No response from API', 0, $e);
            }
        }
        $responseDate = new DateTime();

        $content = $response->getBody()->getContents();

        $logData = new LogData($requestId, $groupId, $sequenceId++, $requestData, $content, $requestDate, $responseDate);

        $this->gatewayLogger->forAll(
            function (GatewayLogger $responseLogger) use ($logData) {
                $responseLogger->log($logData);
            }
        );

        $isValid = $this->gatewayConfig->getRequestEnginePublicKey()->verify(
            $content,
            $response->getHeader('PSP-Certificate-Signature')[0],
            $response->getHeader('PSP-Certificate-Thumbprint')[0]
        );

        if (!$isValid) {
            $this->logger->error('OpenSSLException: Response headers are not valid');

            throw new OpenSSLException('Response headers are not valid');
        }

        try {
            $decoder = new SimpleDecoder();

            return $decoder->decode($content);
        } catch (DecodeException $e) {
            $this->logger->error('DecodeException: Cannot decode json');
            throw new ApiGatewayException('Cannot decode json', 0, $e);
        } catch (InvalidPropertyAccessException $e) {
            throw new ApiGatewayException('Cannot access property in json data', 0, $e);
        }
    }
}
