<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\CommandBus\ExecutePayment;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\CannotExecutePaymentException;
use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\Repository\PaymentRepository;
use IssetBV\TalosBundle\Entity\TalosPaymentIssuer;
use IssetBV\TalosBundle\Entity\TalosPaymentMethod;
use IssetBV\TalosBundle\Gateway\ApiGateway;
use IssetBV\TalosBundle\Gateway\Request\ClientData;
use IssetBV\TalosBundle\Gateway\Request\Request;
use IssetBV\TalosBundle\Gateway\Request\Service;
use IssetBV\TalosBundle\Gateway\Shared\GroupField;
use IssetBV\TalosBundle\Gateway\Shared\SingleField;
use IssetBV\TalosBundle\ResponseHandler\PaymentResponseHandler;
use Money\Formatter\DecimalMoneyFormatter;
use Money\MoneyFormatter;
use PhpOption\Some;

/**
 * Class TalosCreatePaymentHandler.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class ExecutePaymentHandler
{
    /**
     * @var MoneyFormatter
     */
    private $formatter;

    /**
     * @var ApiGateway
     */
    private $gateway;

    /**
     * @var PaymentRepository
     */
    private $paymentRepository;

    /**
     * @var PaymentResponseHandler
     */
    private $paymentResponseHandler;

    /**
     * TalosCreatePaymentHandler constructor.
     *
     * @param ApiGateway $gateway
     * @param DecimalMoneyFormatter $moneyFormatter
     */
    public function __construct(ApiGateway $gateway, DecimalMoneyFormatter $moneyFormatter)
    {
        $this->gateway = $gateway;
        $this->formatter = $moneyFormatter;
    }

    /**
     * @param ExecutePaymentCommand $command
     *
     * @throws CannotExecutePaymentException
     */
    public function handle(ExecutePaymentCommand $command)
    {
        $paymentMethod = $this->findPaymentMethod($command);

        if ($paymentMethod->getServiceName() === 'iDEAL') {
            $request = Some::ensure($this->createIdealPaymentRequest(
                $this->findPayment($command->getPaymentIdentifier()),
                $command->getClientData()->getOrElse(ClientData::empty())
            ));
        } else {
            throw new CannotExecutePaymentException(self::class . ' cannot handle ' . $paymentMethod->getServiceName() . ' at the moment');
        }

        // Send the request through the gateway
        $request->forAll(function (Request $request) {
            $this->gateway->send([$request], 'Web');
        });
    }

    /**
     * @param ExecutablePayment $payment
     * @param ClientData $clientData
     *
     * @throws CannotExecutePaymentException
     *
     * @return Request
     */
    protected function createIdealPaymentRequest(ExecutablePayment $payment, ClientData $clientData): Request
    {
        $amount = $this->formatter->format($payment->getAmount());

        /** @var TalosPaymentIssuer $issuer */
        $issuer = $payment->getPaymentIssuer()
            ->getOrThrow(new CannotExecutePaymentException());

        if (!$issuer instanceof TalosPaymentIssuer) {
            throw new CannotExecutePaymentException('Issuer must by of type ' . TalosPaymentIssuer::class . '. ' . get_class($issuer) . ' given');
        }

        $request = new Request($clientData, [
            new Service('iDEAL', 'Pay', [
                new GroupField('Payment', [
                    new SingleField('InvoiceNumber', $payment->getInvoice()->getNumber()->getAsString()),
                    new SingleField('Reference', $payment->getReference()),
                    new SingleField('Description', 'Payment for Friday'),
                    new SingleField('Currency', $payment->getAmount()->getCurrency()->getCode()),
//                    new SingleField('RecurrentType', 'OneOff'),
                    new SingleField('RecurrentType', $payment->getRecurrentType()),
                    new SingleField('AmountDebit', $amount),
                ]),
                new SingleField('Issuer', $issuer->getValue()),
            ]),
            new Service('WalletSystem', 'Transfer', [
                new SingleField('SourceWalletKey', $payment->getPaymentMethod()->getWallet()->getIdentifier()),
                new SingleField('DestinationWalletKey', $payment->getWallet()->getIdentifier()),
                new SingleField('Amount', $amount),
                new SingleField('Description', 'Payment for Friday'),
                new GroupField('Payment', [
//                    new SingleField('PaymentKey', 'PAYREF123'),
                    new SingleField('WaitForPaymentSuccess', 'true'), // because ideal is fast as shit
                ]),
            ]),
        ], $payment->getReturnUrl());

        $request->addResponseHandler($this->paymentResponseHandler);

        return $request;
    }

    private function findPayment($paymentIdentifier): ExecutablePayment
    {
        return $this->paymentRepository
            ->findOneByRemoteIdentifier($paymentIdentifier)
            ->getOrThrow(new CannotExecutePaymentException('Because the payment doesnt exist'));
    }

    /**
     * @param ExecutePaymentCommand $command
     *
     * @throws CannotExecutePaymentException
     *
     * @return TalosPaymentMethod
     */
    private function findPaymentMethod(ExecutePaymentCommand $command): TalosPaymentMethod
    {
        $paymentMethod = $this
            ->findPayment($command->getPaymentIdentifier())
            ->getPaymentMethod();

        if ($paymentMethod instanceof TalosPaymentMethod) {
            return $paymentMethod;
        }

        throw new CannotExecutePaymentException(self::class . ' cannot handle instances of ' . get_class($paymentMethod));
    }
}
