<?php

declare(strict_types=1);

namespace IssetBV\TalosBundle\Cli;

use IssetBV\PaymentBundle\CommandBus\ExecutePayment\ExecutePaymentCommand;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\Repository\PaymentIssuerRepository;
use IssetBV\PaymentBundle\Domain\Wallet;
use IssetBV\PaymentBundle\Factory\InvoiceFactory;
use IssetBV\PaymentBundle\Factory\PaymentFactory;
use League\Tactician\CommandBus;
use Money\Exception\ParserException;
use Money\MoneyParser;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class CreatePaymentCommand.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
class CreatePaymentCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var InvoiceFactory
     */
    private $invoiceFactory;

    /**
     * @var MoneyParser
     */
    private $moneyParser;

    /**
     * @var PaymentFactory
     */
    private $paymentFactory;

    /**
     * @var PaymentIssuerRepository
     */
    private $paymentIssuerRepository;

    public function __construct(CommandBus $commandBus, MoneyParser $moneyParser, PaymentIssuerRepository $paymentIssuerRepository)
    {
        parent::__construct('talos:payment:create');

        $this->commandBus = $commandBus;
        $this->moneyParser = $moneyParser;
        $this->paymentIssuerRepository = $paymentIssuerRepository;
    }

    protected function configure()
    {
        $this->addArgument('wallet', InputArgument::REQUIRED, 'The remote identifier of the wallet');
        $this->addArgument('amount', InputArgument::REQUIRED, ' The amount of money you want to add');
        $this->addArgument('type', InputArgument::OPTIONAL, 'OneOff, First or Recurring', 'OneOff');
        $this->addArgument('issuer', InputArgument::OPTIONAL, 'The issuer code', 'ABNANL2A');
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @throws InvalidArgumentException
     * @throws ParserException
     *
     * @return int|null|void
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $amount = $this->moneyParser->parse($input->getArgument('amount'), 'EUR');
        $issuer = $this->findIssuer($input);

        $invoice = $this->invoiceFactory->createInvoice();
        $payment = $this->paymentFactory->createPayment($invoice, $amount, $issuer->getPaymentMethod(), $issuer, $input->getArgument('type'));
        //@too this wont really work because we're not linking the wallet

        $this->commandBus->handle(new ExecutePaymentCommand($payment->getIdentifier()));

//        $output->writeln('Pay here: <info>' . $payment->getPaymentUrl() . '</info>');
    }

    private function findIssuer(InputInterface $input): PaymentIssuer
    {
        $issuer = $this->paymentIssuerRepository->findOneByCode($input->getArgument('issuer'))
            ->getOrThrow(new \InvalidArgumentException('Issuer not found'));

        return $issuer;
    }
}
