<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Repository;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\EntityRepository;
use IssetBV\PaymentBundle\Domain\Identifier\LocalPaymentIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionIdentifier;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use PhpOption\Option;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class DoctrineSubscriptionRepository extends EntityRepository implements SubscriptionRepository
{
    /**
     * @return Collection|Subscription[]
     */
    public function findAll()
    {
        return $this->matching(Criteria::create());
    }

    public function optionallyFind(SubscriptionIdentifier $identifier): Option
    {
        return Option::ensure(parent::find($identifier->asString()));
    }

    public function findByPayment(LocalPaymentIdentifier $paymentIdentifier): Option
    {
        $qb = $this->createQueryBuilder('subscription');
        $qb->join('subscription.subscriptionTerms', 'term');
        $qb->join('term.payment', 'payment');
        $qb->where($qb->expr()->eq('payment', ':payment'));
        $qb->setParameter('payment', $paymentIdentifier->asString());

        return Option::ensure($qb->getQuery()->getOneOrNullResult());
    }

    public function findByOriginalPayment(LocalPaymentIdentifier $paymentIdentifier): Option
    {
        return Option::fromValue($this->findOneBy(['originalPayment' => $paymentIdentifier->asString()]));
    }

    /**
     * @return Subscription[]
     */
    public function findPendingSubscriptions()
    {
        return $this->findSubscriptionsWithStatus(SubscriptionStatus::pending());
    }

    /**
     * Returns subscriptions that have expired but are also valid.
     *
     * @return Subscription[]
     */
    public function findExpiredAndValidSubscriptions()
    {
        $qb = $this->createQueryBuilder('subscription');
        $qb->where($qb->expr()->andX(
            $qb->expr()->lte('subscription.denormalizedExpirationDate', ':today'),
            $qb->expr()->lte('subscription.denormalizedStatus', ':status')
        ));

        $qb->setParameter('today', new \DateTime());
        $qb->setParameter('status', SubscriptionStatus::valid()->getCode());

        return $qb->getQuery()->getResult();
    }

    /**
     * @param SubscriptionStatus $subscriptionStatus
     *
     * @return Subscription[]
     */
    private function findSubscriptionsWithStatus(SubscriptionStatus $subscriptionStatus)
    {
        $criteria = Criteria::create();
        $criteria->where(Criteria::expr()->eq('denormalizedStatus', $subscriptionStatus->getCode()));

        /** @var Subscription[] $subscriptions */
        $subscriptions = $this->matching($criteria);

        return $subscriptions;
    }
}
