<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Factory;

use IssetBV\PaymentBundle\Domain\ExecutablePayment;
use IssetBV\PaymentBundle\Domain\Invoice\Invoice;
use IssetBV\PaymentBundle\Domain\Payment;
use IssetBV\PaymentBundle\Domain\PaymentIssuer;
use IssetBV\PaymentBundle\Domain\PaymentMethod;
use Money\Money;

/**
 * This describes some type of class that can create payment objects. Payment objects may have different
 * properties depending on the project. The payment system relies on this interface to create payments when needed.
 *
 * @author Tim Fennis <tim@isset.nl>
 */
interface PaymentFactory
{
    /**
     * Create a new Payment Entity.
     *
     * @param Invoice $invoice The invoice that should be linked to the payment
     * @param Money $money The value of the payment (the amount of money the payment is for)
     * @param PaymentMethod $paymentMethod The payment method to use
     * @param PaymentIssuer|null $issuer The payment issuer to use (in case you're creating an iDeal payment)
     * @param Payment\PaymentType $type The type of the payment @see Payment\PaymentType
     *
     * @return Payment
     */
    public function createPayment(
        Invoice $invoice,
        Money $money,
        PaymentMethod $paymentMethod,
        PaymentIssuer $issuer = null,
        Payment\PaymentType $type
    ): Payment;

    /**
     * Create a new ExecutablePayment Entity.
     *
     * @param Invoice $invoice The invoice that should be linked to the payment
     * @param Money $money The value of the payment (the amount of money the payment is for)
     * @param PaymentMethod $paymentMethod The payment method to use
     * @param PaymentIssuer|null $issuer The payment issuer to use (in case you're creating an iDeal payment)
     * @param Payment\PaymentType $type The type of the payment @see Payment\PaymentType
     *
     * @return ExecutablePayment A payment that can be executed by the talos gateway
     */
    public function createExecutablePayment(
        Invoice $invoice,
        Money $money,
        PaymentMethod $paymentMethod,
        PaymentIssuer $issuer = null,
        Payment\PaymentType $type
    ): ExecutablePayment;

    /**
     * Create a new ExecutablePayment Entity.
     *
     * @param Invoice $invoice The invoice that should be linked to the payment
     * @param Money $amount The value of the payment (the amount of money the payment is for)
     * @param ExecutablePayment $originalPayment The payment this recurring payment should be based on. (Must be type First)
     *
     * @return ExecutablePayment A payment that can be executed by the talos gateway
     */
    public function createRecurringPayment(
        Invoice $invoice,
        Money $amount,
        ExecutablePayment $originalPayment
    ): ExecutablePayment;
}
