<?php

declare(strict_types=1);

namespace IssetBV\PaymentBundle\Cli;

use Doctrine\Common\Collections\Criteria;
use Exception;
use IssetBV\PaymentBundle\CommandBus\DenormalizeSubscriptionStatus\DenormalizeSubscriptionStatusCommand;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\CannotRenewSubscriptionException;
use IssetBV\PaymentBundle\CommandBus\RenewSubscription\RenewSubscriptionCommand;
use IssetBV\PaymentBundle\Domain\Subscription\Repository\SubscriptionRepository;
use IssetBV\PaymentBundle\Domain\Subscription\Subscription;
use IssetBV\PaymentBundle\Domain\Subscription\SubscriptionStatus;
use League\Tactician\CommandBus;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use function Functional\each;
use function Functional\filter;
use function Functional\map;

/**
 * @author Tim Fennis <tim@isset.nl>
 */
class RenewSubscriptionsCommand extends Command
{
    /**
     * @var CommandBus
     */
    private $commandBus;

    /**
     * @var SymfonyStyle
     */
    private $console;

    /**
     * @var SubscriptionRepository
     */
    private $subscriptionRepository;

    public function __construct(
        CommandBus $commandBus,
        SubscriptionRepository $subscriptionRepository
    ) {
        parent::__construct('subscription:renewAll');
        $this->commandBus = $commandBus;
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function configure()
    {
        // do nothing for now
        $this->addOption('force', 'f', InputOption::VALUE_NONE, 'Unless you specify this option no payments will be executed');
        $this->addOption('migration-mode', 'm', InputOption::VALUE_NONE, 'Run this command in migration mode also updating all outdated subscriptions');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->console = new SymfonyStyle($input, $output);

        if ($input->getOption('migration-mode')) {
            $criteria = Criteria::create();
            $criteria->where(Criteria::expr()->isNull('denormalizedExpirationDate'));

            $this->subscriptionRepository->matching($criteria);
            $this->console->note('Running in migration mode');
            $this->console->note('Searching and updating subscriptions without denormalized expirationDate...');

            /** @var Subscription[] $subscriptions */
            $subscriptions = $this->subscriptionRepository->matching($criteria);
            $this->updateSubscriptions($subscriptions);
        }

        $this->console->text('Searching and updating pending subscriptions...');
        $this->updateSubscriptions($this->subscriptionRepository->findPendingSubscriptions());

        $this->console->text('Searching and updating expired subscriptions...');
        $this->updateSubscriptions($this->subscriptionRepository->findExpiredAndValidSubscriptions());

        //@todo check this
        $this->console->section('Renewing subscriptions');
        $this->renewSubscriptions(!$input->getOption('force'));
    }

    /**
     * @param Subscription[] $subscriptions
     */
    private function updateSubscriptions($subscriptions)
    {
        $numberOfSubscriptions = count($subscriptions);

        if (0 === $numberOfSubscriptions) {
            $this->console->success('Nothing to update');

            return;
        }

        $this->console->progressStart($numberOfSubscriptions);

        try {
            foreach ($subscriptions as $subscription) {
                try {
                    $this->commandBus->handle(new DenormalizeSubscriptionStatusCommand($subscription->getId()));
                } catch (CannotRenewSubscriptionException $e) {
                    $this->console->warning($e->getMessage());
                }

                $this->console->progressAdvance(1);
            }

            $this->console->progressFinish();
            $this->console->success('Subscriptions updated');
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());
        }
    }

    /**
     * @param mixed $mock
     *
     * @throws Exception
     */
    private function renewSubscriptions($mock = true)
    {
        $this->console->text('Searching for subscriptions that are both expired and not canceled');

        $criteria = Criteria::create();
        $criteria->andWhere(Criteria::expr()->eq('denormalizedStatus', SubscriptionStatus::expired()->getCode()));
        $criteria->andWhere(Criteria::expr()->eq('dateCanceled', null));

        $subscriptions = $this->subscriptionRepository->matching($criteria);
        $this->console->text(sprintf('Found %d subscriptions matching criteria', $subscriptions->count()));
        $this->console->text('Filtering subscriptions that are not renewable');

        $subscriptions = filter($subscriptions, function (Subscription $subscription) {
            return $subscription->getOriginalPayment()->isDefined() &&
                $subscription->findRenewablePayment()->isDefined();
        });

        $this->console->text(sprintf('Found %d subscriptions', count($subscriptions)));

        try {
            $subscriptions = filter($subscriptions, function (Subscription $subscription) {
                return $subscription->getOriginalPayment()->isDefined();
            });

            $this->console->listing(map($subscriptions, function (Subscription $subscription) {
                return sprintf('[mock] Would renew subscription: %s', $subscription->getId()->asString());
            }));

            if (false === $mock) {
                $this->console->progressStart(count($subscriptions));

                each($subscriptions, function (Subscription $subscription) {
                    try {
                        $this->commandBus->handle(new RenewSubscriptionCommand($subscription->getId()));
                    } catch (CannotRenewSubscriptionException $e) {
                        $this->console->warning($e->getMessage());
                    }
                    $this->console->progressAdvance(1);
                });

                $this->console->progressFinish();
                $this->console->success('Subscriptions updated');
            }
        } catch (Exception $e) {
            $this->console->progressFinish();
            $this->console->caution('Exception: ' . $e->getMessage());

            throw $e;
        }
    }
}
