<?php

namespace IssetBV\SettingsBundle\Setting;

use Doctrine\ORM\EntityManagerInterface;
use IssetBV\SettingsBundle\Converter\ConverterInterface;

class Setting
{

    /**
     * @var \Doctrine\ORM\EntityManagerInterface
     */
    private $entityManager;

    /**
     * @var \IssetBv\JobBundle\Repository\SettingRepository
     */
    private $settingRepo;

    /**
     *
     * @var \Doctrine\ORM\EntityRepository
     */
    private $settingTypeRepo;

    /**
     * @var \IssetBV\SettingsBundle\Converter\ConverterInterface
     */
    private $converter;

    public function __construct(EntityManagerInterface $entityManager, ConverterInterface $converter)
    {
        $this->entityManager   = $entityManager;
        $this->converter       = $converter;
        $this->settingRepo     = $entityManager->getRepository("IssetBVSettingsBundle:Setting");
        $this->settingTypeRepo = $entityManager->getRepository("IssetBVSettingsBundle:SettingType");
    }

    /**
     * Get a setting
     *
     * @param String $setting
     * @param array $group
     * @return mixed 
     * @throws \IssetBV\SettingsBundle\Exception\SettingMissingException
     * @throws \IssetBV\SettingsBundle\Exception\ConverterMissingException
     */
    public function get($setting, array $group = null)
    {
        $settingData = $this->settingRepo->getSetting($setting, $group);
        return $this->converter->load($settingData->getType()->getType(), $settingData->getValue());
    }

    /**
     * Set a setting
     *
     * @param String $setting
     * @param mixed $value
     * @param String $type
     * @param array $group
     * @param bool $autoFlush
     * @throws \IssetBV\SettingsBundle\Exception\TypeNotFoundException
     * @throws \IssetBV\SettingsBundle\Exception\ConverterMissingException
     */
    public function set($setting, $value, $type = 'string', array $group = null, $autoFlush = true)
    {
        //Get the type data if it doesn't exists we throw an exception
        $typeData = $this->settingTypeRepo->findOneBy(array('type' => $type));
        if (is_null($typeData)) {
            throw new \IssetBV\SettingsBundle\Exception\TypeNotFoundException("Type not found: " . $type);
        }

        //Prepare $value data for save
        $value       = $this->converter->save($type, $value);
        //check if the setting allready exists if not we make a new entity else we set the update time
        $settingData = $this->settingRepo->findOneBy(array('key' => $setting, 'group' => null));

        if (is_null($settingData)) {
            $settingData = new \IssetBV\SettingsBundle\Entity\Setting();
            $settingData->setKey($setting);
            $settingData->setCreated(new \DateTime());
            $this->entityManager->persist($settingData);
        } else {
            $settingData->setUpdated(new \DateTime());
        }

        $settingData->setType($typeData);
        $settingData->setValue($value);

        if ($autoFlush) {
            $this->entityManager->flush();
        }
    }

}