<?php

namespace IssetBv\TpcRestClientBundle\Services\File;

use \IssetBv\TpcRestClientBundle\Client\ClientFactory;
use \IssetBv\TpcRestClientBundle\Client\Payload;

class FileHandler
{

    /**
     * @var \IssetBv\TpcRestClientBundle\Client\RestClient
     */
    private $client;
    protected $servers       = false;
    protected $companyHeader = 'company';

    public function __construct($url)
    {
        $this->client = ClientFactory::getClient($url);
    }

    public function setCompany($company)
    {
        $this->company = $company;
    }

    public function recover($company, $identifier, $callback, $logIdentifier = false)
    {
        $payload = new Payload('file/' . $identifier . '/recover', 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('callback', $callback);
        if ($logIdentifier) {
            $payload->addData('logIdentifier', $logIdentifier);
        }
        return $this->client->sendPayload($payload);
    }

    public function deleteFiles(array $identifiers)
    {
        $payload = new Payload('delete/file/' . implode(',', $identifiers));
        return $this->client->sendPayload($payload);
    }

    public function updateFile($company, $identifier, array $settings)
    {
        $payload = new Payload('file/' . $identifier, 'post');
        $payload->setHeader($this->companyHeader, $company);
        foreach ($settings as $name => $settings) {
            $payload->addData($name, $settings);
        }
        return $this->client->sendPayload($payload);
    }

    public function addFile($company, $location, $server, $type, $failbackIdentifier = false, $setLive = true, $removeable = true, $removeable_from = false)
    {
        if (!$this->checkServer($server)) {
            throw new \Exception();
        }

        if ($failbackIdentifier && !$this->checkIdentifier($company, $failbackIdentifier)) {
            throw new \Exception();
        }

        $payload = new Payload('file/add', 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('location', $location);
        $payload->addData('server', $server);
        $payload->addData('type', $type);
        $payload->addData('live', $setLive);
        $payload->addData('removeable', $removeable);
        if ($removeable_from) {
            $payload->addData('removeable_from', $removeable_from);
        }


        if ($failbackIdentifier) {
            $payload->addData('failback_identifier', $failbackIdentifier);
        }

        return $this->client->sendPayload($payload);
    }

    public function lock($company, $identifier, $callerId, $callback)
    {
        $payload = new Payload('lock/' . $identifier . '/' . $callerId, 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('callback', $callback);
        try {
            $data = $this->client->sendPayload($payload);
            return $data->status;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function unLock($company, $identifier, $callerId, $status)
    {
        $payload = new Payload('/lock/' . $identifier . '/release', 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('status', $status);
        $payload->addData('caller_id', $callerId);
        try {
            $data = $this->client->sendPayload($payload);
            return $data->status;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getUuid($company, $location, $server, $type)
    {
        $payload = new Payload('file/uuid', 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('location', $location);
        $payload->addData('server', $server);
        $payload->addData('type', $type);
        try {
            $data = $this->client->sendPayload($payload);
            return $data->uuid;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function checkIdentifier($company, $identifier)
    {
        $data = $this->getByIdentifier($company, $identifier);
        return !empty($data);
    }

    public function getByIdentifier($company, $identifier)
    {
        $payload = new Payload('file/' . $identifier);
        $payload->setHeader($this->companyHeader, $company);
        return $this->client->sendPayload($payload);
    }

    public function createGroup($company, $identifier, array $files)
    {
        $payload = new Payload('group/add', 'post');
        $payload->setHeader($this->companyHeader, $company);
        $payload->addData('identifier', $identifier);


        foreach ($files as $key => $fileId) {
            $payload->addData("files[$key]", $fileId);
        }

        try {
            return $this->client->sendPayload($payload);
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getGroup($company, $identifier)
    {
        $payload = new Payload('group/' . $identifier, 'get');
        $payload->setHeader($this->companyHeader, $company);
        try {
            return $this->client->sendPayload($payload);
        } catch (\Exception $e) {
            return false;
        }
    }

    public function checkServer($server)
    {
        if (!$this->servers) {
            $this->servers = $this->getServerList();
        }
        return isset($this->servers->$server);
    }

    public function getServerList()
    {
        $payload = new Payload('server/list');
        return $this->client->sendPayload($payload);
    }

}