<?php

declare(strict_types=1);

namespace IssetBV\Queuing\MessageBundle\DependencyInjection;

use IssetBV\Queuing\Message\Consumer\ConsumerContainer;
use IssetBV\Queuing\Message\Decoder\DecoderContainer;
use IssetBV\Queuing\Message\DeQueuer\DeQueuer;
use IssetBV\Queuing\Message\DeQueuer\DeQueuerExceptionHandlerInterface;
use IssetBV\Queuing\Message\DeQueuer\DeQueuerInterface;
use IssetBV\Queuing\Message\Encoder\EncoderContainer;
use IssetBV\Queuing\Message\Queuer\Queuer;
use IssetBV\Queuing\Message\Queuer\QueuerInterface;
use IssetBV\Queuing\MessageBundle\Command\QueuingDeQueueCommand;
use IssetBV\Queuing\MessageBundle\Command\QueuingDeQueueContinuesCommand;
use IssetBV\Queuing\MessageBundle\Command\QueuingQueueWarmupCommand;
use IssetBV\Queuing\MessageBundle\Service\DeQueuer\Consume\DoctrineClearPreConsume;
use IssetBV\Queuing\MessageBundle\Service\DeQueuer\DeQueuerExceptionHandlerReject;
use IssetBV\Queuing\Queue\ConnectionInterface;
use IssetBV\Queuing\Queue\QueueProxy;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\HttpKernel\Kernel;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class IssetBVQueuingMessageExtension extends Extension
{
    const PREFIX = 'isset_bv_queuing_message.';
    const ENCODER_CONTAINER_DEFAULT = self::PREFIX . 'encoder.container.default';
    const DECODER_CONTAINER_DEFAULT = self::PREFIX . 'decoder.container.default';
    const CONSUMER_CONTAINER_DEFAULT = self::PREFIX . 'consumer.container.default';
    const DEQUEUER_EXCEPTION_HANDLER_DEFAULT = self::PREFIX . 'dequeuer.exception_handler.default';

    /**
     * @var ContainerBuilder
     */
    private $container;
    private $defaultQueue;

    /**
     * {@inheritdoc}
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $this->container = $container;

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $this->defaultQueue = $config['default_queue'];
        $this->container->register(DoctrineClearPreConsume::class)->setAutowired(true);

        $this->buildDefaultEncoder($config);
        $this->buildDecoderContainer($config);
        $this->buildConsumerContainer($config);
        $this->buildDeQueuerExceptionHandler();

        if (array_key_exists('queues', $config)) {
            $this->buildQueues($config['queues']);
        }
        $this->buildCommands();
    }

    private function buildCommands()
    {
        $definition = $this->container->register(QueuingDeQueueCommand::class);
        $definition->setAutowired(true);
        $definition->setLazy(true);
        $definition->addTag('console.command', ['command' => QueuingDeQueueCommand::NAME]);

        $definition = $this->container->register(QueuingDeQueueContinuesCommand::class);
        $definition->setAutowired(true);
        $definition->setLazy(true);
        $definition->addTag('console.command', ['command' => QueuingDeQueueContinuesCommand::NAME]);

        $definition = $this->container->register(QueuingQueueWarmupCommand::class);
        $definition->setAutowired(true);
        $definition->setLazy(true);
        $definition->addTag('console.command', ['command' => QueuingQueueWarmupCommand::NAME]);
    }

    /**
     * @param $config
     */
    private function buildDefaultEncoder(array $config)
    {
        $encoderDefault = $this->container->register(
            self::ENCODER_CONTAINER_DEFAULT,
            EncoderContainer::class
        );
        $encoderDefault->setLazy(true);

        if (null !== $config['default_encoder']) {
            $encoderDefault->addArgument(new Reference($config['default_encoder']));
        }
    }

    private function buildDecoderContainer(array $config)
    {
        $decoderContainer = $this->container->register(
            self::DECODER_CONTAINER_DEFAULT,
            DecoderContainer::class
        );
        $decoderContainer->setLazy(true);
        if (null !== $config['default_decoder']) {
            $decoderContainer->addArgument(new Reference($config['default_decoder']));
        }
    }

    private function buildConsumerContainer(array $config)
    {
        $consumerContainerDefault = $this->container->register(
            self::CONSUMER_CONTAINER_DEFAULT,
            ConsumerContainer::class
        );
        $consumerContainerDefault->setLazy(true);
    }

    private function buildQueues(array $queues)
    {
        foreach ($queues as $name => $queue) {
            $this->buildQueue($name, $queue);
        }
    }

    private function buildQueue(string $name, array $settings)
    {
        $proxyClassName = self::PREFIX . 'queue.' . $name;
        $proxyClass = $this->container->register(
            $proxyClassName,
            QueueProxy::class
        );
        $proxyClass->setPublic(true);
        $proxyClass->addTag(self::PREFIX . 'queue');
        $proxyClass->setLazy(true);

        $proxyClass->addArgument(new Reference(ConnectionInterface::CONNECTION_PREFIX . $settings['connection']));
        $proxyClass->addArgument($settings['exchange']);
        $proxyClass->addArgument($settings['queue'] ?? $settings['exchange']);
        $proxyClass->addArgument($settings['options']);

        $this->buildQueuer($name, $proxyClassName);
        $this->buildDequeuer($name, $proxyClassName, $settings);
    }

    private function buildDequeuer(string $name, string $queueClassName, array $settings)
    {
        $dequeuerClassName = self::PREFIX . 'dequeuer.' . $name;
        $dequeuerClass = $this->container->register(
            $dequeuerClassName,
            DeQueuer::class
        );

        $dequeuerClass->setPublic(true);
        $dequeuerClass->setLazy(true);

        $decoderContainer = $settings['decoder_container'] ?? self::DECODER_CONTAINER_DEFAULT;
        $handlerContainer = $settings['consumer_container'] ?? self::CONSUMER_CONTAINER_DEFAULT;
        $exceptionHandler = $settings['exception_handler'] ?? self::DEQUEUER_EXCEPTION_HANDLER_DEFAULT;

        $this->addConsumesToContainer($handlerContainer, $settings);

        $dequeuerClass->addArgument(new Reference($decoderContainer));
        $dequeuerClass->addArgument(new Reference($handlerContainer));
        $dequeuerClass->addArgument(new Reference($queueClassName));
        $dequeuerClass->addArgument(new Reference($exceptionHandler));
        if ($this->defaultQueue === $name) {
            if (Kernel::MAJOR_VERSION < 3 || (Kernel::MAJOR_VERSION === 3 && Kernel::MINOR_VERSION < 3)) {
                $dequeuerClass->addAutowiringType(DeQueuerInterface::class);
            }
            $this->container->setAlias(DeQueuerInterface::class, $dequeuerClassName);
        }
    }

    private function addConsumesToContainer(string $handlerContainer, array $settings)
    {
        $handlerContainerDefinition = $this->container->getDefinition($handlerContainer);

        if ($settings['consumer_container_clear_doctrine']) {
            $handlerContainerDefinition->addMethodCall('addPreConsume', [new Reference(DoctrineClearPreConsume::class)]);
        }

        foreach ($settings['consumer_container_pre_consumes'] as $preConsume) {
            $handlerContainerDefinition->addMethodCall('addPreConsume', [new Reference($preConsume)]);
        }

        foreach ($settings['consumer_container_post_consumes'] as $postConsume) {
            $handlerContainerDefinition->addMethodCall('addPostConsume', [new Reference($postConsume)]);
        }
    }

    private function buildQueuer(string $name, string $queueClassName)
    {
        $queuerClassName = self::PREFIX . 'queuer.' . $name;
        $queuerClass = $this->container->register(
            $queuerClassName,
            Queuer::class
        );
        $queuerClass->setPublic(true);
        $queuerClass->setLazy(true);
        $encoderContainer = $settings['encoder_container'] ?? self::ENCODER_CONTAINER_DEFAULT;
        $queuerClass->addArgument(new Reference($encoderContainer));
        $queuerClass->addArgument(new Reference($queueClassName));
        if ($this->defaultQueue === $name) {
            if (Kernel::MAJOR_VERSION < 3 || (Kernel::MAJOR_VERSION === 3 && Kernel::MINOR_VERSION < 3)) {
                $queuerClass->addAutowiringType(QueuerInterface::class);
            }
            $this->container->setAlias(QueuerInterface::class, $queuerClassName);
        }
    }

    private function buildDeQueuerExceptionHandler()
    {
        $class = $this->container->register(
            self::DEQUEUER_EXCEPTION_HANDLER_DEFAULT,
            DeQueuerExceptionHandlerReject::class
        );
        $class->setLazy(true);
        $class->addMethodCall('setLogger', [new Reference('logger')]);
        if (Kernel::MAJOR_VERSION < 3 || (Kernel::MAJOR_VERSION === 3 && Kernel::MINOR_VERSION < 3)) {
            $class->addAutowiringType(DeQueuerExceptionHandlerInterface::class);
        }
        $this->container->setAlias(DeQueuerExceptionHandlerInterface::class, self::DEQUEUER_EXCEPTION_HANDLER_DEFAULT);
    }
}
