<?php

declare(strict_types=1);

namespace IssetBV\Queuing\MessageBundle\Command;

use IssetBV\Queuing\Message\DeQueuer\DeQueuerInterface;
use IssetBV\Queuing\MessageBundle\DependencyInjection\IssetBVQueuingMessageExtension;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class QueuingDeQueueContinuesCommand extends ContainerAwareCommand
{
    const NAME = 'issetbv:queuing:dequeue:continues';

    protected function configure()
    {
        $this
            ->setName(self::NAME)
            ->setDescription('de queues continues for the given queue or the default queue')
            ->addArgument('queue', InputArgument::OPTIONAL)
            ->addOption('timeout', 'ti', InputOption::VALUE_OPTIONAL, 'timeout time in ms', 1000)
            ->addOption('wait_time_no_message_ms', 'wt', InputOption::VALUE_OPTIONAL, 'Wait time if there are no messages on the queue',
                100)
            ->addOption('message_limit', 'ml', InputOption::VALUE_OPTIONAL, 'maximum amount of messages to process default is endless')
            ->addOption('time_limit', 'tl', InputOption::VALUE_OPTIONAL, 'maximum amount of time(seconds) default is 10 minutes', 600)
            ->addOption('kill_switch', 'ks', InputOption::VALUE_OPTIONAL, 'file to watch as kill switch', null)
            ->addOption('warmup', 'w', InputOption::VALUE_OPTIONAL, 'warmup the queue on startup. only works if queue is given', true);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $timeout = (int) $input->getOption('timeout');
        $messageLimit = $input->getOption('message_limit');
        $waitTimeNoMessageMS = (int) $input->getOption('wait_time_no_message_ms');
        $timeLimit = (int) $input->getOption('time_limit');
        $killSwitch = $input->getOption('kill_switch');
        $warmup = (bool) $input->getOption('warmup');

        $queueName = $input->getArgument('queue');
        if (null !== $queueName) {
            $id = IssetBVQueuingMessageExtension::PREFIX . 'dequeuer.' . $queueName;
            if ($warmup) {
                $this->warmup($queueName);
            }
        } else {
            $id = DeQueuerInterface::class;
        }

        $deQueuer = $this->getContainer()->get($id);
        /* @var $deQueuer DeQueuerInterface */
        $start = time();
        while (0 < $messageLimit || null === $messageLimit) {
            $messagesReceivedAmount = $deQueuer->dequeue($timeout)->amountReceived();

            if (null !== $messageLimit) {
                $messageLimit -= $messagesReceivedAmount;
                if ($messageLimit <= 0) {
                    break;
                }
            }

            if ((time() - $start) >= $timeLimit) {
                break;
            }

            if (null !== $killSwitch && file_exists($killSwitch)) {
                unlink($killSwitch);
                break;
            }

            if ($waitTimeNoMessageMS > 0 && 0 === $messagesReceivedAmount) {
                usleep($waitTimeNoMessageMS * 1000);
            }
        }
    }

    private function warmup(string $queueName)
    {
        $queue = IssetBVQueuingMessageExtension::PREFIX . 'queue.' . $queueName;
        $this->getContainer()->get($queue)->warmUp(['type' => null]);
    }
}
