<?php

declare(strict_types=1);

namespace IssetBV\Queuing\MessageBundle\Command;

use IssetBV\Queuing\Message\DeQueuer\DeQueuerInterface;
use IssetBV\Queuing\Message\DeQueuer\StatisticInterface;
use IssetBV\Queuing\MessageBundle\DependencyInjection\IssetBVQueuingMessageExtension;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class QueuingDeQueueContinuesCommand extends ContainerAwareCommand
{
    const NAME = 'issetbv:queuing:dequeue:continues';

    protected function configure()
    {
        $this
            ->setName(self::NAME)
            ->addArgument('consumer', InputArgument::OPTIONAL)
            ->setDescription('de queues continues for the given queue or the default queue')
            ->addOption('timeout', 'ti', InputOption::VALUE_OPTIONAL, 'timeout time in ms', 1000)
            ->addOption('wait_time_no_message_ms', 'wt', InputOption::VALUE_OPTIONAL, 'Wait time if there are no messages on the queue', 0)
            ->addOption('message_limit', 'ml', InputOption::VALUE_OPTIONAL, 'maximum amount of messages to process default is endless')
            ->addOption('time_limit', 'tl', InputOption::VALUE_OPTIONAL, 'maximum amount of time(seconds) default is 10 minutes', 36000)
            ->addOption('kill_switch', 'ks', InputOption::VALUE_OPTIONAL, 'file to watch as kill switch', null);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $timeout = (int) $input->getOption('timeout');
        $messageLimit = $input->getOption('message_limit');
        $waitTimeNoMessageMS = (int) $input->getOption('wait_time_no_message_ms');
        $timeLimit = (int) $input->getOption('time_limit');
        $killSwitch = $input->getOption('kill_switch');

        if (null !== $input->getArgument('consumer')) {
            $id = IssetBVQueuingMessageExtension::PREFIX . 'dequeuer.' . $input->getArgument('consumer');
        } else {
            $id = DeQueuerInterface::class;
        }
        $deQueuer = $this->getContainer()->get($id);
        /* @var $deQueuer DeQueuerInterface */
        $start = time();
        while (0 !== $messageLimit) {
            $count = $this->dequeue($deQueuer, $timeout, $waitTimeNoMessageMS)->amountReceived();

            if (null !== $messageLimit) {
                $messageLimit -= $count;
            }

            if ((time() - $start) >= $timeLimit) {
                break;
            }

            if (null !== $killSwitch && file_exists($killSwitch)) {
                unlink($killSwitch);
                break;
            }
        }
    }

    private function dequeue(DeQueuerInterface $deQueuer, int $timeout, int $waitTimeNoMessageMS): StatisticInterface
    {
        $statistic = $deQueuer->dequeue($timeout);
        if ($waitTimeNoMessageMS > 0 && 0 === $statistic->amountReceived()) {
            usleep($waitTimeNoMessageMS * 1000);
        }

        return $statistic;
    }
}
