<?php

declare(strict_types=1);

namespace IssetBV\Queuing\MessageBundle\DependencyInjection;

use IssetBV\Queuing\Message\Consumer\ConsumerContainer;
use IssetBV\Queuing\Message\Decoder\DecoderContainer;
use IssetBV\Queuing\Message\DeQueuer\DeQueuer;
use IssetBV\Queuing\Message\DeQueuer\DeQueuerInterface;
use IssetBV\Queuing\Message\Encoder\EncoderContainer;
use IssetBV\Queuing\Message\Queuer\Queuer;
use IssetBV\Queuing\Message\Queuer\QueuerInterface;
use IssetBV\Queuing\Queue\ConnectionInterface;
use IssetBV\Queuing\Queue\QueueProxy;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\HttpKernel\Kernel;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * @see http://symfony.com/doc/current/cookbook/bundles/extension.html
 */
class IssetBVQueuingMessageExtension extends Extension
{
    const PREFIX = 'isset_bv_queuing_message.';
    const ENCODER_CONTAINER_DEFAULT = self::PREFIX . 'encoder.container.default';
    const DECODER_CONTAINER_DEFAULT = self::PREFIX . 'decoder.container.default';
    const CONSUMER_CONTAINER_DEFAULT = self::PREFIX . 'consumer.container.default';

    /**
     * @var ContainerBuilder
     */
    private $container;
    private $defaultQueue;

    /**
     * {@inheritdoc}
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $this->container = $container;

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $this->defaultQueue = $config['default_queue'];

        $this->buildDefaultEncoder($config);
        $this->buildDecoderContainer($config);
        $this->buildConsumerContainer($config);

        if (array_key_exists('queues', $config)) {
            $this->buildQueues($config['queues']);
        }
    }

    /**
     * @param $config
     */
    private function buildDefaultEncoder(array $config)
    {
        $encoderDefault = $this->container->register(
            self::ENCODER_CONTAINER_DEFAULT,
            EncoderContainer::class
        );
        $encoderDefault->setLazy(true);

        if (null !== $config['default_encoder']) {
            $encoderDefault->addArgument(new Reference($config['default_encoder']));
        }
    }

    private function buildDecoderContainer(array $config)
    {
        $decoderContainer = $this->container->register(
            self::DECODER_CONTAINER_DEFAULT,
            DecoderContainer::class
        );
        $decoderContainer->setLazy(true);
        if (null !== $config['default_decoder']) {
            $decoderContainer->addArgument(new Reference($config['default_decoder']));
        }
    }

    private function buildConsumerContainer(array $config)
    {
        $consumerContainerDefault = $this->container->register(
            self::CONSUMER_CONTAINER_DEFAULT,
            ConsumerContainer::class
        );
        $consumerContainerDefault->setLazy(true);
    }

    private function buildQueues(array $queues)
    {
        foreach ($queues as $name => $queue) {
            $this->buildQueue($name, $queue);
        }
    }

    private function buildQueue(string $name, array $settings)
    {
        $proxyClassName = self::PREFIX . 'queue.' . $name;
        $proxyClass = $this->container->register(
            $proxyClassName,
            QueueProxy::class
        );
        $proxyClass->setLazy(true);

        $proxyClass->addArgument(new Reference(ConnectionInterface::CONNECTION_PREFIX . $settings['connection']));
        $proxyClass->addArgument($settings['exchange']);
        $proxyClass->addArgument($settings['queue'] ?? $settings['exchange']);

        $this->buildQueuer($name, $proxyClassName);
        $this->buildDequeuer($name, $proxyClassName);
    }

    private function buildDequeuer(string $name, string $queueClassName)
    {
        $dequeuerClassName = self::PREFIX . 'dequeuer.' . $name;
        $dequeuerClass = $this->container->register(
            $dequeuerClassName,
            DeQueuer::class
        );
        $dequeuerClass->setLazy(true);
        $decoderContainer = $settings['decoder_container'] ?? self::DECODER_CONTAINER_DEFAULT;
        $handlerContainer = $settings['consumer_container'] ?? self::CONSUMER_CONTAINER_DEFAULT;

        $dequeuerClass->addArgument(new Reference($decoderContainer));
        $dequeuerClass->addArgument(new Reference($handlerContainer));
        $dequeuerClass->addArgument(new Reference($queueClassName));
        if ($this->defaultQueue === $name) {
            if (Kernel::MAJOR_VERSION < 3 || (Kernel::MAJOR_VERSION === 3 && Kernel::MINOR_VERSION < 3)) {
                $dequeuerClass->addAutowiringType(DeQueuerInterface::class);
            }
            $this->container->setAlias(DeQueuerInterface::class, $dequeuerClassName);
        }
    }

    private function buildQueuer(string $name, string $queueClassName)
    {
        $queuerClassName = self::PREFIX . 'queuer.' . $name;
        $queuerClass = $this->container->register(
            $queuerClassName,
            Queuer::class
        );
        $queuerClass->setLazy(true);
        $encoderContainer = $settings['encoder_container'] ?? self::ENCODER_CONTAINER_DEFAULT;
        $queuerClass->addArgument(new Reference($encoderContainer));
        $queuerClass->addArgument(new Reference($queueClassName));
        if ($this->defaultQueue === $name) {
            if (Kernel::MAJOR_VERSION < 3 || (Kernel::MAJOR_VERSION === 3 && Kernel::MINOR_VERSION < 3)) {
                $queuerClass->addAutowiringType(QueuerInterface::class);
            }
            $this->container->setAlias(QueuerInterface::class, $queuerClassName);
        }
    }
}
