<?php

declare(strict_types=1);

namespace IssetBV\Queuing\Message\Consumer;

use IssetBV\Queuing\Message\Consumer\Exception\ConsumerNotFoundException;
use IssetBV\Queuing\Message\MessageInterface;

class ConsumerContainer implements ConsumerContainerInterface
{
    /**
     * @var ConsumerInterface[]
     */
    private $consumers = [];
    /**
     * @var PreConsumeInterface[]
     */
    private $preConsumes = [];
    /**
     * @var PostConsumeInterface[]
     */
    private $postConsumes = [];

    public function add(ConsumerInterface $consumer)
    {
        $this->consumers[] = $consumer;
    }

    public function addPreConsume(PreConsumeInterface $preConsume)
    {
        if (!\in_array($preConsume, $this->preConsumes)) {
            $this->preConsumes[] = $preConsume;
        }
    }

    public function addPostConsume(PostConsumeInterface $postConsume)
    {
        if (!\in_array($postConsume, $this->postConsumes)) {
            $this->postConsumes[] = $postConsume;
        }
    }

    /**
     * @param MessageInterface $message
     *
     * @throws ConsumerNotFoundException
     *
     * @return ResponseInterface
     */
    public function consume(MessageInterface $message): ResponseInterface
    {
        $consumersUsed = $this->fetchUsedConsumers($message);

        $count = \count($consumersUsed);
        if (0 === $count) {
            throw new ConsumerNotFoundException('Consumer not found');
        }

        if (1 !== $count) {
            $consumersUsed = $this->sortConsumers($consumersUsed);
        }

        foreach ($this->preConsumes as $preConsume) {
            $preConsume->preConsume($message);
        }

        $response = new ResponseAggregate();
        foreach ($consumersUsed as $consumer) {
            $consumerResponse = $consumer->consume($message);
            $response->addResponse($consumerResponse);
            if ($consumerResponse->stopPropagation()) {
                break;
            }
        }

        foreach ($this->postConsumes as $postConsume) {
            $postConsume->postConsume($message, $response);
        }

        return $response;
    }

    /**
     * @param ConsumerInterface[] $consumers
     *
     * @return ConsumerInterface[]
     */
    private function sortConsumers(array $consumers): array
    {
        usort($consumers, function (ConsumerInterface $a, ConsumerInterface $b) {
            $aWeight = $this->getWeight($a);
            $bWeight = $this->getWeight($b);
            if ($aWeight > $bWeight) {
                return -1;
            }

            if ($aWeight < $bWeight) {
                return 1;
            }

            return 0;
        });

        return $consumers;
    }

    private function getWeight(ConsumerInterface $consumer)
    {
        if ($consumer instanceof ConsumerWeightedInterface) {
            return $consumer->getWeight();
        }

        return 300;
    }

    /**
     * @param MessageInterface $message
     *
     * @return ConsumerInterface[]
     */
    private function fetchUsedConsumers(MessageInterface $message): array
    {
        $consumersUsed = [];
        foreach ($this->consumers as $consumer) {
            if ($consumer->consumes($message)) {
                $consumersUsed[] = $consumer;
            }
        }

        return $consumersUsed;
    }
}
