<?php

declare(strict_types=1);

namespace IssetBV\Queuing\Message\Consumer;

use IssetBV\Queuing\Message\Consumer\Exception\ConsumerNotFoundException;
use IssetBV\Queuing\Message\MessageInterface;

class ConsumerContainer implements ConsumerContainerInterface
{
    /**
     * @var ConsumerInterface[]
     */
    private $consumers = [];

    /**
     * @param ConsumerInterface $consumer
     */
    public function add(ConsumerInterface $consumer)
    {
        $this->consumers[] = $consumer;
    }

    /**
     * @param MessageInterface $message
     *
     * @throws ConsumerNotFoundException
     *
     * @return ResponseInterface
     */
    public function consume(MessageInterface $message): ResponseInterface
    {
        $consumersUsed = $this->fetchUsedConsumers($message);

        $count = \count($consumersUsed);
        if (0 === $count) {
            throw new ConsumerNotFoundException('Consumer not found');
        }

        if (1 !== $count) {
            $consumersUsed = $this->sortConsumers($consumersUsed);
        }

        $response = new ResponseAggregate();
        foreach ($consumersUsed as $consumer) {
            $consumerResponse = $consumer->consume($message);
            $response->addResponse($consumerResponse);
            if ($consumerResponse->stopPropagation()) {
                break;
            }
        }

        return $response;
    }

    /**
     * @param ConsumerInterface[] $consumers
     *
     * @return ConsumerInterface[]
     */
    private function sortConsumers(array $consumers): array
    {
        usort($consumers, function (ConsumerInterface $a, ConsumerInterface $b) {
            $aWeight = $this->getWeight($a);
            $bWeight = $this->getWeight($b);
            if ($aWeight > $bWeight) {
                return -1;
            }

            if ($aWeight < $bWeight) {
                return 1;
            }

            return 0;
        });

        return $consumers;
    }

    private function getWeight(ConsumerInterface $consumer)
    {
        if ($consumer instanceof ConsumerWeightedInterface) {
            return $consumer->getWeight();
        }

        return 300;
    }

    /**
     * @param MessageInterface $message
     *
     * @return ConsumerInterface[]
     */
    private function fetchUsedConsumers(MessageInterface $message): array
    {
        $consumersUsed = [];
        foreach ($this->consumers as $consumer) {
            if ($consumer->consumes($message)) {
                $consumersUsed[] = $consumer;
            }
        }

        return $consumersUsed;
    }
}
